@extends('layouts.admin')

@section('title', 'Assignment Management')

@section('content')
<div class="breadcrumb d-flex flex-wrap align-items-center justify-content-between gap-3 mb-24">
    <div class="">
        <h1 class="fw-semibold mb-4 h6 text-primary-light">Assignment Management</h1>
        <div class="">
            <a href="{{ route('admin.dashboard') }}" class="text-secondary-light hover-text-primary hover-underline">Dashboard</a>
            <span class="text-secondary-light">/ Assignment</span>
        </div>
    </div>
    <a href="{{ route('admin.homework.create') }}" class="btn btn-primary-600 d-flex align-items-center gap-6">
        <span class="d-flex text-md">
            <i class="ri-add-large-line"></i>
        </span>
        Add Assignment
    </a>
</div>

@if(session('success'))
<div class="alert alert-success alert-dismissible fade show" role="alert">
    {{ session('success') }}
    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
</div>
@endif

<div class="mt-24">
    <div class="card h-100">
        <div class="card-header border-bottom bg-base py-16 px-24 d-flex align-items-center justify-content-between">
            <h6 class="text-lg fw-semibold mb-0">All Assignments</h6>
        </div>
        <div class="card-body p-20">
            @if($homework->count())
                <div class="table-responsive">
                    <table class="table align-middle mb-0">
                        <thead>
                            <tr>
                                <th>Title</th>
                                <th>Teacher</th>
                                <th>Class / Section</th>
                                <th>Subject</th>
                                <th>Assigned Date</th>
                                <th>Due Date</th>
                                <th>Status</th>
                                <th class="text-end">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($homework as $hw)
                            <tr>
                                <td class="fw-semibold text-primary-light">{{ $hw->title }}</td>
                                <td>{{ $hw->teacher->name ?? '-' }}</td>
                                <td>
                                    {{ $hw->schoolClass->name ?? '-' }}
                                    @if($hw->section)
                                        / {{ $hw->section->name }}
                                    @endif
                                </td>
                                <td>{{ $hw->subject->name ?? '-' }}</td>
                                <td>{{ $hw->assigned_date->format('d M Y') }}</td>
                                <td>{{ $hw->due_date->format('d M Y') }}</td>
                                <td>
                                    @php
                                        $statusClass = 'bg-secondary-subtle text-secondary';
                                        if($hw->status === 'active') $statusClass = 'bg-success-subtle text-success';
                                        elseif($hw->status === 'completed') $statusClass = 'bg-primary-subtle text-primary';
                                        elseif($hw->status === 'cancelled') $statusClass = 'bg-danger-subtle text-danger';
                                    @endphp
                                    <span class="badge {{ $statusClass }} mb-1 d-block">{{ ucfirst($hw->status) }}</span>
                                    <small class="text-secondary-light">
                                        Submissions: {{ $hw->submissions->count() }}
                                    </small>
                                </td>
                                <td class="text-end">
                                    <div class="d-flex gap-2 justify-content-end">
                                        <button type="button" class="btn btn-sm btn-outline-success" style="width: 32px; height: 32px; padding: 0; display: flex; align-items: center; justify-content: center;" onclick="openMarkSubmissionsModal({{ $hw->id }})" title="Mark Submissions">
                                            <i class="ri-checkbox-line"></i>
                                        </button>
                                        <a href="{{ route('admin.homework.show', $hw->id) }}" class="btn btn-sm btn-outline-primary" style="width: 32px; height: 32px; padding: 0; display: flex; align-items: center; justify-content: center;" title="View">
                                            <i class="ri-eye-line"></i>
                                        </a>
                                        <a href="{{ route('admin.homework.edit', $hw->id) }}" class="btn btn-sm btn-outline-secondary" style="width: 32px; height: 32px; padding: 0; display: flex; align-items: center; justify-content: center;" title="Edit">
                                            <i class="ri-pencil-line"></i>
                                        </a>
                                        <form action="{{ route('admin.homework.destroy', $hw->id) }}" method="POST" class="d-inline" onsubmit="return confirm('Are you sure you want to delete this assignment?');">
                                            @csrf
                                            @method('DELETE')
                                            <button type="submit" class="btn btn-sm btn-outline-danger" style="width: 32px; height: 32px; padding: 0; display: flex; align-items: center; justify-content: center;" title="Delete">
                                                <i class="ri-delete-bin-line"></i>
                                            </button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            @else
                <p class="mb-0 text-secondary-light text-center py-20">No assignments found.</p>
            @endif
        </div>
    </div>
</div>

<!-- Mark Submissions Modal -->
<div class="modal fade" id="markSubmissionsModal" tabindex="-1" aria-labelledby="markSubmissionsModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-xl">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="markSubmissionsModalLabel">Mark Assignment Submissions</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form id="markSubmissionsForm" method="POST">
                @csrf
                <div class="modal-body">
                    <div id="modalHomeworkInfo" class="mb-3 p-3 bg-light rounded">
                        <h6 class="mb-1" id="modalHomeworkTitle"></h6>
                        <small class="text-secondary" id="modalHomeworkDueDate"></small>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label fw-semibold">Submission Date <span class="text-danger">*</span></label>
                        <input type="date" name="submitted_date" id="submission_date" class="form-control" required value="{{ now()->toDateString() }}">
                    </div>

                    <div class="mb-3">
                        <div class="d-flex justify-content-between align-items-center mb-2">
                            <label class="form-label fw-semibold mb-0">Select Students</label>
                            <div>
                                <button type="button" class="btn btn-sm btn-outline-primary" onclick="selectAll()">Select All</button>
                                <button type="button" class="btn btn-sm btn-outline-secondary" onclick="deselectAll()">Deselect All</button>
                            </div>
                        </div>
                        <div id="studentsList" style="max-height: 500px; overflow-y: auto; border: 1px solid #dee2e6; border-radius: 0.375rem; padding: 10px;">
                            <div class="text-center py-3">
                                <div class="spinner-border spinner-border-sm" role="status">
                                    <span class="visually-hidden">Loading...</span>
                                </div>
                                <p class="mt-2 mb-0 text-secondary">Loading students...</p>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Mark Submissions</button>
                </div>
            </form>
        </div>
    </div>
</div>

@endsection

@section('scripts')
<script>
    let currentHomeworkId = null;

    function openMarkSubmissionsModal(homeworkId) {
        currentHomeworkId = homeworkId;
        const modal = new bootstrap.Modal(document.getElementById('markSubmissionsModal'));
        modal.show();
        
        // Reset form
        document.getElementById('markSubmissionsForm').action = '{{ route("admin.homework.mark-submissions", ":id") }}'.replace(':id', homeworkId);
        document.getElementById('studentsList').innerHTML = '<div class="text-center py-3"><div class="spinner-border spinner-border-sm" role="status"><span class="visually-hidden">Loading...</span></div><p class="mt-2 mb-0 text-secondary">Loading students...</p></div>';
        
        // Fetch students
        fetch(`{{ route('admin.homework.students', ':id') }}`.replace(':id', homeworkId))
            .then(response => response.json())
            .then(data => {
                document.getElementById('modalHomeworkTitle').textContent = data.homework.title;
                document.getElementById('modalHomeworkDueDate').textContent = 'Due Date: ' + new Date(data.homework.due_date).toLocaleDateString();
                
                let studentsHtml = '';
                if (data.students.length === 0) {
                    studentsHtml = '<p class="text-center text-secondary py-3">No students found for this assignment.</p>';
                } else {
                    data.students.forEach(student => {
                        const checked = student.submitted ? 'checked' : '';
                        let statusHtml = '';
                        if (student.submitted) {
                            if (student.is_late) {
                                statusHtml = '<span class="badge bg-danger-subtle text-danger d-block mb-1">Late Submission</span>';
                            } else {
                                statusHtml = '<span class="badge bg-success-subtle text-success d-block mb-1">Already Submitted</span>';
                            }
                            if (student.submitted_date) {
                                statusHtml += '<small class="text-secondary d-block">' + student.submitted_date + '</small>';
                            }
                        }
                        studentsHtml += `
                            <div class="mb-3 p-3 border rounded">
                                <div class="form-check mb-2">
                                    <input class="form-check-input student-checkbox" type="checkbox" name="student_ids[]" value="${student.id}" id="student_${student.id}" ${checked}>
                                    <label class="form-check-label w-100" for="student_${student.id}">
                                        <div class="d-flex justify-content-between align-items-center">
                                            <div>
                                                <strong>${student.name}</strong>
                                                <small class="text-secondary d-block">${student.admission_no} - ${student.class}${student.section ? ' / ' + student.section : ''}</small>
                                            </div>
                                            <div class="text-end">
                                                ${statusHtml}
                                            </div>
                                        </div>
                                    </label>
                                </div>
                                <div class="row g-2 mt-2">
                                    <div class="col-md-4">
                                        <label class="form-label small mb-1">Grade</label>
                                        <input type="text" name="grades[${student.id}]" class="form-control form-control-sm" placeholder="A, B, C..." value="${student.grade || ''}" maxlength="10">
                                    </div>
                                    <div class="col-md-8">
                                        <label class="form-label small mb-1">Teacher Notes</label>
                                        <textarea name="teacher_notes[${student.id}]" class="form-control form-control-sm" rows="2" placeholder="Add notes about this submission...">${student.teacher_notes || ''}</textarea>
                                    </div>
                                </div>
                            </div>
                        `;
                    });
                }
                document.getElementById('studentsList').innerHTML = studentsHtml;
            })
            .catch(error => {
                console.error('Error:', error);
                document.getElementById('studentsList').innerHTML = '<p class="text-center text-danger py-3">Error loading students. Please try again.</p>';
            });
    }

    function selectAll() {
        document.querySelectorAll('.student-checkbox').forEach(checkbox => {
            checkbox.checked = true;
        });
    }

    function deselectAll() {
        document.querySelectorAll('.student-checkbox').forEach(checkbox => {
            checkbox.checked = false;
        });
    }

    // Handle form submission
    document.getElementById('markSubmissionsForm').addEventListener('submit', function(e) {
        const checkedBoxes = document.querySelectorAll('.student-checkbox:checked');
        if (checkedBoxes.length === 0) {
            e.preventDefault();
            alert('Please select at least one student.');
            return false;
        }
    });
</script>
@endsection
