<?php

namespace App\Services;

use App\Models\DeviceToken;
use App\Models\Notification;
use App\Models\User;
use App\Models\Student;
use App\Models\Setting;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Cache;

class PushNotificationService
{
    protected $projectId;
    protected $serviceAccountPath;
    protected $pushEnabled = true;
    protected $fcmUrl;

    public function __construct()
    {
        $setting = Setting::first();
        $this->pushEnabled = $setting->push_notifications_enabled ?? true;
        $this->projectId = $setting->fcm_project_id ?? config('services.fcm.project_id');
        $this->serviceAccountPath = storage_path('app/private/firebase-service-account.json');
        $this->fcmUrl = "https://fcm.googleapis.com/v1/projects/{$this->projectId}/messages:send";
    }

    /**
     * Get OAuth2 access token for FCM v1 API
     */
    protected function getAccessToken()
    {
        // Cache the token for 50 minutes (tokens expire in 60 minutes)
        return Cache::remember('fcm_access_token', 50 * 60, function () {
            if (!file_exists($this->serviceAccountPath)) {
                Log::error('Firebase service account file not found: ' . $this->serviceAccountPath);
                return null;
            }

            $serviceAccount = json_decode(file_get_contents($this->serviceAccountPath), true);
            
            if (!$serviceAccount) {
                Log::error('Invalid Firebase service account JSON');
                return null;
            }

            // Create JWT
            $header = base64_encode(json_encode(['alg' => 'RS256', 'typ' => 'JWT']));
            $now = time();
            $claims = [
                'iss' => $serviceAccount['client_email'],
                'scope' => 'https://www.googleapis.com/auth/firebase.messaging',
                'aud' => 'https://oauth2.googleapis.com/token',
                'iat' => $now,
                'exp' => $now + 3600,
            ];
            $payload = base64_encode(json_encode($claims));
            
            // Sign with private key
            $signatureInput = str_replace(['+', '/', '='], ['-', '_', ''], $header) . '.' . 
                              str_replace(['+', '/', '='], ['-', '_', ''], $payload);
            
            openssl_sign($signatureInput, $signature, $serviceAccount['private_key'], 'SHA256');
            $signature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
            
            $jwt = $signatureInput . '.' . $signature;

            // Exchange JWT for access token
            $response = Http::asForm()->post('https://oauth2.googleapis.com/token', [
                'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
                'assertion' => $jwt,
            ]);

            if ($response->successful()) {
                return $response->json()['access_token'];
            }

            Log::error('Failed to get FCM access token: ' . $response->body());
            return null;
        });
    }

    /**
     * Send push notification to a single token using FCM v1 API
     */
    protected function sendToToken(string $token, string $title, string $body, array $data = [])
    {
        $accessToken = $this->getAccessToken();
        
        if (!$accessToken) {
            Log::info('Push notification skipped: Could not get access token');
            return false;
        }

        $message = [
            'message' => [
                'token' => $token,
                'notification' => [
                    'title' => $title,
                    'body' => $body,
                ],
                'data' => array_map('strval', $data), // FCM v1 requires string values
                'android' => [
                    'priority' => 'high',
                    'notification' => [
                        'sound' => 'default',
                        'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
                    ],
                ],
                'apns' => [
                    'payload' => [
                        'aps' => [
                            'sound' => 'default',
                        ],
                    ],
                ],
            ],
        ];

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $accessToken,
                'Content-Type' => 'application/json',
            ])->post($this->fcmUrl, $message);

            if ($response->successful()) {
                Log::info('FCM v1 notification sent successfully');
                return true;
            } else {
                Log::error('FCM v1 Error: ' . $response->body());
                
                // If token is invalid, mark it as inactive
                if ($response->status() === 404 || str_contains($response->body(), 'UNREGISTERED')) {
                    DeviceToken::where('token', $token)->update(['is_active' => false]);
                }
                return false;
            }
        } catch (\Exception $e) {
            Log::error('FCM v1 Exception: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Send push notification to specific tokens
     */
    public function sendToTokens(array $tokens, string $title, string $body, array $data = [])
    {
        if (!$this->pushEnabled) {
            Log::info('Push notification skipped: Push notifications are disabled');
            return false;
        }

        if (empty($tokens) || empty($this->projectId)) {
            Log::info('Push notification skipped: No tokens or FCM project not configured');
            return false;
        }

        $success = false;
        foreach ($tokens as $token) {
            if ($this->sendToToken($token, $title, $body, $data)) {
                $success = true;
            }
        }

        return $success;
    }

    /**
     * Send push notification to a specific user
     */
    public function sendToUser(int $userId, string $title, string $body, array $data = [])
    {
        $tokens = DeviceToken::getActiveTokensForUser($userId);
        return $this->sendToTokens($tokens, $title, $body, $data);
    }

    /**
     * Send push notification to multiple users
     */
    public function sendToUsers(array $userIds, string $title, string $body, array $data = [])
    {
        $tokens = DeviceToken::getActiveTokensForUsers($userIds);
        return $this->sendToTokens($tokens, $title, $body, $data);
    }

    /**
     * Send push notification to parent of a student
     */
    public function sendToStudentParent(int $studentId, string $title, string $body, array $data = [])
    {
        $student = Student::find($studentId);
        if ($student && $student->parent_id) {
            return $this->sendToUser($student->parent_id, $title, $body, $data);
        }
        return false;
    }

    /**
     * Send push notification to all parents
     */
    public function sendToAllParents(string $title, string $body, array $data = [])
    {
        $parentIds = User::where('role', 'parent')->pluck('id')->toArray();
        return $this->sendToUsers($parentIds, $title, $body, $data);
    }

    /**
     * Send push notification to parents of students in a class
     */
    public function sendToClassParents(int $classId, string $title, string $body, array $data = [])
    {
        $parentIds = Student::where('class_id', $classId)
            ->whereNotNull('parent_id')
            ->pluck('parent_id')
            ->unique()
            ->toArray();
        
        return $this->sendToUsers($parentIds, $title, $body, $data);
    }

    // ============= Event-specific notifications =============

    /**
     * Fee Generated Notification
     */
    public function notifyFeeGenerated($feeCollection)
    {
        $student = $feeCollection->student;
        if (!$student || !$student->parent_id) return false;

        $title = 'New Fee Generated';
        $body = "A new fee of ₹{$feeCollection->amount} has been generated for {$student->name}";

        // Also create in-app notification
        $this->createInAppNotification(
            'fee',
            $title,
            $body,
            null,
            $student->id,
            null,
            'fee',
            $feeCollection->id,
            'fees'
        );

        return $this->sendToUser($student->parent_id, $title, $body, [
            'type' => 'fee',
            'fee_id' => $feeCollection->id,
            'student_id' => $student->id,
        ]);
    }

    /**
     * Fee Payment Reminder
     */
    public function notifyFeeReminder($feeCollection)
    {
        $student = $feeCollection->student;
        if (!$student || !$student->parent_id) return false;

        $title = 'Payment Reminder';
        $body = "Reminder: Fee of ₹{$feeCollection->due_amount} is pending for {$student->name}";

        return $this->sendToUser($student->parent_id, $title, $body, [
            'type' => 'fee_reminder',
            'fee_id' => $feeCollection->id,
        ]);
    }

    /**
     * Payment Received Notification
     */
    public function notifyPaymentReceived($feeCollection, $amount = null)
    {
        $student = $feeCollection->student;
        if (!$student || !$student->parent_id) return false;

        $paidAmount = $amount ?? $feeCollection->paid_amount;
        $title = 'Payment Received';
        $body = "Payment of ₹{$paidAmount} received for {$student->name}. Thank you!";

        // Create in-app notification
        $this->createInAppNotification(
            'Payment Received',
            $body,
            'fee',
            null,
            $student->id,
            null
        );

        return $this->sendToUser($student->parent_id, $title, $body, [
            'type' => 'payment_received',
            'fee_id' => (string) $feeCollection->id,
        ]);
    }

    /**
     * Exam Scheduled Notification
     */
    public function notifyExamScheduled($examSchedule)
    {
        $title = 'Exam Scheduled';
        $subjectName = $examSchedule->subject->name ?? 'Subject';
        $examDate = $examSchedule->exam_date->format('d M Y');
        $body = "{$subjectName} exam scheduled for {$examDate}";

        // Create in-app notification for class
        $this->createInAppNotification(
            'exam',
            $title,
            $body,
            null,
            null,
            $examSchedule->class_id,
            'exam',
            $examSchedule->id,
            'exams'
        );

        return $this->sendToClassParents($examSchedule->class_id, $title, $body, [
            'type' => 'exam',
            'exam_id' => $examSchedule->id,
        ]);
    }

    /**
     * Result Published Notification
     */
    public function notifyResultPublished($examResult)
    {
        $student = $examResult->student;
        if (!$student || !$student->parent_id) return false;

        $subjectName = $examResult->examSchedule->subject->name ?? 'Subject';
        $title = 'Result Published';
        $body = "{$subjectName} result is now available for {$student->name}. Marks: {$examResult->marks_obtained}";

        // Create in-app notification
        $this->createInAppNotification(
            'result',
            $title,
            $body,
            $student->parent_id,
            $student->id,
            null,
            'result',
            $examResult->id,
            'exams'
        );

        return $this->sendToUser($student->parent_id, $title, $body, [
            'type' => 'result',
            'result_id' => $examResult->id,
        ]);
    }

    /**
     * Leave Status Update Notification
     */
    public function notifyLeaveStatusUpdate($leaveRequest)
    {
        $student = $leaveRequest->student;
        if (!$student || !$student->parent_id) return false;

        $status = ucfirst($leaveRequest->status);
        $title = "Leave Request {$status}";
        
        // Handle date formatting safely
        $fromDate = $leaveRequest->from_date ? $leaveRequest->from_date->format('d M') : 'N/A';
        $toDate = $leaveRequest->to_date ? $leaveRequest->to_date->format('d M') : 'N/A';
        $body = "Your leave request for {$student->name} from {$fromDate} to {$toDate} has been {$leaveRequest->status}";

        // Create in-app notification
        $this->createInAppNotification(
            'leave',
            $title,
            $body,
            $student->parent_id,
            $student->id,
            null,
            'leave',
            $leaveRequest->id,
            'leave'
        );

        return $this->sendToUser($student->parent_id, $title, $body, [
            'type' => 'leave',
            'leave_id' => $leaveRequest->id,
            'status' => $leaveRequest->status,
        ]);
    }

    /**
     * Homework Assigned Notification
     */
    public function notifyHomeworkAssigned($homework)
    {
        $title = 'New Homework Assigned';
        $body = "{$homework->title} - Due: {$homework->due_date->format('d M Y')}";

        // Create in-app notification for class
        $this->createInAppNotification(
            'homework',
            $title,
            $body,
            null,
            null,
            $homework->class_id,
            'homework',
            $homework->id,
            'homework'
        );

        return $this->sendToClassParents($homework->class_id, $title, $body, [
            'type' => 'homework',
            'homework_id' => $homework->id,
        ]);
    }

    /**
     * Exam Result Notification
     */
    public function notifyExamResult($examSchedule, $student, $result)
    {
        if (!$student || !$student->parent_id) return false;

        $examName = $examSchedule->examType->name ?? 'Exam';
        $subjectName = $examSchedule->subject->name ?? 'Subject';
        $maxMarks = $examSchedule->max_marks ?? 100;
        $marksObtained = $result->marks_obtained ?? 0;
        $percentage = $maxMarks > 0 ? round(($marksObtained / $maxMarks) * 100, 1) : 0;

        $title = 'Exam Result Available';
        $body = "{$student->name}: {$subjectName} ({$examName}) - {$marksObtained}/{$maxMarks} ({$percentage}%)";

        // Create in-app notification
        $this->createInAppNotification(
            'exam',
            $title,
            $body,
            $student->parent_id,
            $student->id,
            null,
            'result',
            $result->id,
            'exam'
        );

        return $this->sendToUser($student->parent_id, $title, $body, [
            'type' => 'exam_result',
            'exam_schedule_id' => (string) $examSchedule->id,
            'result_id' => (string) $result->id,
            'student_id' => (string) $student->id,
        ]);
    }

    /**
     * Attendance Alert (Absent)
     */
    public function notifyAttendanceAbsent($attendance)
    {
        $student = $attendance->student;
        if (!$student || !$student->parent_id) return false;

        $title = 'Attendance Alert';
        $body = "{$student->name} was marked absent today ({$attendance->date->format('d M Y')})";

        return $this->sendToUser($student->parent_id, $title, $body, [
            'type' => 'attendance',
            'attendance_id' => $attendance->id,
        ]);
    }

    /**
     * General Announcement
     */
    public function notifyAnnouncement($note, $forAllParents = true, $classId = null)
    {
        $title = $note->title;
        $body = $note->body ?? $note->title;

        // Create in-app notification
        $this->createInAppNotification(
            'announcement',
            $title,
            $body,
            null,
            null,
            $forAllParents ? null : $classId,
            'note',
            $note->id,
            null,
            $forAllParents
        );

        if ($forAllParents) {
            return $this->sendToAllParents($title, $body, [
                'type' => 'announcement',
                'note_id' => $note->id,
            ]);
        } elseif ($classId) {
            return $this->sendToClassParents($classId, $title, $body, [
                'type' => 'announcement',
                'note_id' => $note->id,
            ]);
        }

        return false;
    }

    /**
     * Create in-app notification
     */
    protected function createInAppNotification(
        $type,
        $title,
        $message,
        $parentId = null,
        $studentId = null,
        $classId = null,
        $referenceType = null,
        $referenceId = null,
        $actionUrl = null,
        $forAllParents = false
    ) {
        $typeConfig = Notification::$types[$type] ?? ['icon' => 'info', 'color' => 'blue'];

        return Notification::create([
            'title' => $title,
            'message' => $message,
            'type' => $type,
            'icon' => $typeConfig['icon'],
            'color' => $typeConfig['color'],
            'for_all_parents' => $forAllParents,
            'parent_id' => $parentId,
            'student_id' => $studentId,
            'class_id' => $classId,
            'reference_type' => $referenceType,
            'reference_id' => $referenceId,
            'action_url' => $actionUrl,
            'is_active' => true,
        ]);
    }
}
