<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class TeacherNotification extends Model
{
    use HasFactory;

    protected $fillable = [
        'teacher_id',
        'type',
        'title',
        'message',
        'icon',
        'icon_bg',
        'link',
        'data',
        'read_at',
    ];

    protected $casts = [
        'data' => 'array',
        'read_at' => 'datetime',
    ];

    /**
     * Scope for unread notifications
     */
    public function scopeUnread($query)
    {
        return $query->whereNull('read_at');
    }

    /**
     * Scope for a specific teacher
     */
    public function scopeForTeacher($query, $teacherId)
    {
        return $query->where('teacher_id', $teacherId);
    }

    /**
     * Mark notification as read
     */
    public function markAsRead()
    {
        $this->update(['read_at' => now()]);
    }

    /**
     * Get the teacher user
     */
    public function teacher()
    {
        return $this->belongsTo(User::class, 'teacher_id');
    }

    /**
     * Create a leave status notification for teacher
     */
    public static function notifyLeaveStatus($leaveRequest)
    {
        $status = ucfirst($leaveRequest->status);
        $fromDate = $leaveRequest->from_date ? $leaveRequest->from_date->format('d M') : 'N/A';
        $toDate = $leaveRequest->to_date ? $leaveRequest->to_date->format('d M') : 'N/A';
        
        return self::create([
            'teacher_id' => $leaveRequest->user_id,
            'type' => 'leave_status',
            'title' => "Leave Request {$status}",
            'message' => "Your leave request from {$fromDate} to {$toDate} has been {$leaveRequest->status}",
            'icon' => 'ri-calendar-check-line',
            'icon_bg' => $leaveRequest->status == 'approved' ? 'bg-success-600' : ($leaveRequest->status == 'rejected' ? 'bg-danger-600' : 'bg-warning-600'),
            'link' => route('teacher.leave-requests.index'),
            'data' => [
                'leave_request_id' => $leaveRequest->id,
            ],
        ]);
    }

    /**
     * Create a homework submission notification for teacher
     */
    public static function notifyHomeworkSubmission($submission, $student, $homework)
    {
        $teacherId = $homework->teacher_id;
        if (!$teacherId) return null;
        
        // Get the teacher's user_id from the teachers table
        $teacher = Teacher::find($teacherId);
        if (!$teacher || !$teacher->user_id) return null;

        return self::create([
            'teacher_id' => $teacher->user_id,
            'type' => 'homework_submission',
            'title' => 'Homework Submitted',
            'message' => "{$student->name} submitted: {$homework->title}",
            'icon' => 'ri-book-open-line',
            'icon_bg' => 'bg-primary-600',
            'link' => route('teacher.homework.show', $homework->id),
            'data' => [
                'homework_id' => $homework->id,
                'submission_id' => $submission->id,
                'student_id' => $student->id,
            ],
        ]);
    }

    /**
     * Create a general notification
     */
    public static function notify($teacherId, $type, $title, $message, $icon = 'ri-notification-3-line', $iconBg = 'bg-info-600', $link = null, $data = [])
    {
        return self::create([
            'teacher_id' => $teacherId,
            'type' => $type,
            'title' => $title,
            'message' => $message,
            'icon' => $icon,
            'icon_bg' => $iconBg,
            'link' => $link,
            'data' => $data,
        ]);
    }
}
