<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Notification extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'message',
        'type',
        'icon',
        'color',
        'for_all_parents',
        'parent_id',
        'student_id',
        'class_id',
        'reference_type',
        'reference_id',
        'action_url',
        'is_active',
        'scheduled_at',
        'expires_at',
        'created_by',
    ];

    protected $casts = [
        'for_all_parents' => 'boolean',
        'is_active' => 'boolean',
        'scheduled_at' => 'datetime',
        'expires_at' => 'datetime',
    ];

    // Notification types with icons and colors
    public static $types = [
        'general' => ['icon' => 'info', 'color' => 'blue'],
        'fee' => ['icon' => 'payment', 'color' => 'orange'],
        'leave' => ['icon' => 'event_busy', 'color' => 'purple'],
        'exam' => ['icon' => 'school', 'color' => 'indigo'],
        'result' => ['icon' => 'assessment', 'color' => 'green'],
        'homework' => ['icon' => 'assignment', 'color' => 'teal'],
        'attendance' => ['icon' => 'fact_check', 'color' => 'amber'],
        'announcement' => ['icon' => 'campaign', 'color' => 'red'],
    ];

    public function parent()
    {
        return $this->belongsTo(User::class, 'parent_id');
    }

    public function student()
    {
        return $this->belongsTo(Student::class);
    }

    public function schoolClass()
    {
        return $this->belongsTo(SchoolClass::class, 'class_id');
    }

    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function reads()
    {
        return $this->hasMany(NotificationRead::class);
    }

    public function isReadBy($userId)
    {
        return $this->reads()->where('user_id', $userId)->exists();
    }

    public function markAsReadBy($userId)
    {
        return NotificationRead::firstOrCreate([
            'notification_id' => $this->id,
            'user_id' => $userId,
        ], [
            'read_at' => now(),
        ]);
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true)
            ->where(function ($q) {
                $q->whereNull('scheduled_at')
                  ->orWhere('scheduled_at', '<=', now());
            })
            ->where(function ($q) {
                $q->whereNull('expires_at')
                  ->orWhere('expires_at', '>', now());
            });
    }

    public function scopeForParent($query, $parentId, $studentIds = [], $classIds = [])
    {
        return $query->where(function ($q) use ($parentId, $studentIds, $classIds) {
            $q->where('for_all_parents', true)
              ->orWhere('parent_id', $parentId)
              ->orWhereIn('student_id', $studentIds)
              ->orWhereIn('class_id', $classIds);
        });
    }
}
