<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class AdminNotification extends Model
{
    use HasFactory;

    protected $fillable = [
        'type',
        'title',
        'message',
        'icon',
        'icon_bg',
        'link',
        'from_user_id',
        'from_role',
        'data',
        'read_at',
    ];

    protected $casts = [
        'data' => 'array',
        'read_at' => 'datetime',
    ];

    /**
     * Scope for unread notifications
     */
    public function scopeUnread($query)
    {
        return $query->whereNull('read_at');
    }

    /**
     * Mark notification as read
     */
    public function markAsRead()
    {
        $this->update(['read_at' => now()]);
    }

    /**
     * Get the user who triggered this notification
     */
    public function fromUser()
    {
        return $this->belongsTo(User::class, 'from_user_id');
    }

    /**
     * Create a leave request notification
     */
    public static function notifyLeaveRequest($leaveRequest, $user, $role)
    {
        $student = $leaveRequest->student;
        $leaveType = $leaveRequest->leaveType;
        
        return self::create([
            'type' => 'leave_request',
            'title' => 'New Leave Request',
            'message' => "{$user->name} ({$role}) submitted a leave request for {$student->name} - {$leaveType->name}",
            'icon' => 'ri-calendar-check-line',
            'icon_bg' => 'bg-warning-600',
            'link' => route('admin.leave-requests.index'),
            'from_user_id' => $user->id,
            'from_role' => $role,
            'data' => [
                'leave_request_id' => $leaveRequest->id,
                'student_id' => $student->id,
                'student_name' => $student->name,
            ],
        ]);
    }

    /**
     * Create a fee payment notification
     */
    public static function notifyFeePayment($feeCollection, $user, $amount)
    {
        $student = $feeCollection->student;
        
        return self::create([
            'type' => 'fee_payment',
            'title' => 'Fee Payment Received',
            'message' => "₹{$amount} payment received for {$student->name}",
            'icon' => 'ri-money-rupee-circle-line',
            'icon_bg' => 'bg-success-600',
            'link' => route('admin.fees-collection.edit', $feeCollection->id),
            'from_user_id' => $user->id ?? null,
            'from_role' => 'parent',
            'data' => [
                'fee_collection_id' => $feeCollection->id,
                'student_id' => $student->id,
                'amount' => $amount,
            ],
        ]);
    }

    /**
     * Create a homework submission notification
     */
    public static function notifyHomeworkSubmission($submission, $student)
    {
        $homework = $submission->homework;
        
        return self::create([
            'type' => 'homework_submission',
            'title' => 'Homework Submitted',
            'message' => "{$student->name} submitted homework: {$homework->title}",
            'icon' => 'ri-book-open-line',
            'icon_bg' => 'bg-primary-600',
            'link' => route('admin.homework.index'),
            'from_user_id' => $student->parent_id,
            'from_role' => 'student',
            'data' => [
                'homework_id' => $homework->id,
                'submission_id' => $submission->id,
                'student_id' => $student->id,
            ],
        ]);
    }

    /**
     * Create a general notification
     */
    public static function notify($type, $title, $message, $icon = 'ri-notification-3-line', $iconBg = 'bg-info-600', $link = null, $userId = null, $role = null, $data = [])
    {
        return self::create([
            'type' => $type,
            'title' => $title,
            'message' => $message,
            'icon' => $icon,
            'icon_bg' => $iconBg,
            'link' => $link,
            'from_user_id' => $userId,
            'from_role' => $role,
            'data' => $data,
        ]);
    }
}
