<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\TeacherNotification;
use Illuminate\Support\Facades\Auth;

class TeacherNotificationController extends Controller
{
    /**
     * Get all notifications for teacher
     */
    public function index()
    {
        $user = Auth::user();
        if (!$user || !$user->isTeacher()) {
            abort(403, 'Unauthorized access');
        }

        $notifications = TeacherNotification::where('teacher_id', $user->id)
            ->orderByDesc('created_at')
            ->paginate(20);

        return view('teacher.notifications.index', compact('notifications', 'user'));
    }

    /**
     * Mark notification as read and redirect
     */
    public function markAsRead($id)
    {
        $user = Auth::user();
        $notification = TeacherNotification::where('teacher_id', $user->id)
            ->findOrFail($id);
        
        $notification->markAsRead();

        if ($notification->link) {
            return redirect($notification->link);
        }

        return redirect()->route('teacher.notifications.index');
    }

    /**
     * Mark all notifications as read
     */
    public function markAllAsRead()
    {
        $user = Auth::user();
        
        TeacherNotification::where('teacher_id', $user->id)
            ->whereNull('read_at')
            ->update(['read_at' => now()]);

        return redirect()->back()->with('success', 'All notifications marked as read');
    }

    /**
     * Delete notification
     */
    public function destroy($id)
    {
        $user = Auth::user();
        $notification = TeacherNotification::where('teacher_id', $user->id)
            ->findOrFail($id);
        
        $notification->delete();

        return redirect()->back()->with('success', 'Notification deleted');
    }
}
