<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Attendance;
use App\Models\SchoolClass;
use App\Models\Section;
use App\Models\Student;
use App\Models\Teacher;
use App\Models\LeaveRequest;

class TeacherAttendanceController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isTeacher()) {
            abort(403, 'Unauthorized access');
        }

        $teacher = Teacher::where('user_id', $user->id)->first();

        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        $sections = Section::where('is_active', true)->orderBy('name')->get();

        $date = $request->input('date', now()->toDateString());
        $classId = $request->input('class_id');
        $sectionId = $request->input('section_id');

        $studentsQuery = Student::with(['schoolClass', 'section'])
            ->where('is_active', true);

        if ($classId) {
            $studentsQuery->where('class_id', $classId);
        }
        if ($sectionId) {
        $studentsQuery->where('section_id', $sectionId);
        }

        $students = $studentsQuery->orderBy('name')->get();

        // Existing attendance for that date
        $existing = Attendance::whereDate('date', $date)
            ->whereIn('student_id', $students->pluck('id'))
            ->get()
            ->keyBy('student_id');

        // Approved leave requests for these students covering the date
        $approvedLeaves = LeaveRequest::where('status', 'approved')
            ->whereNotNull('student_id')
            ->whereDate('from_date', '<=', $date)
            ->whereDate('to_date', '>=', $date)
            ->whereIn('student_id', $students->pluck('id'))
            ->get()
            ->keyBy('student_id');

        return view('teacher.attendance.index', compact(
            'user', 'teacher', 'classes', 'sections', 'students', 'existing', 'approvedLeaves', 'date', 'classId', 'sectionId'
        ));
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isTeacher()) {
            abort(403, 'Unauthorized access');
        }

        $validated = $request->validate([
            'date' => 'required|date',
            'class_id' => 'required|exists:school_classes,id',
            'section_id' => 'required|exists:sections,id',
            'attendance' => 'required|array',
            'attendance.*.status' => 'required|in:present,absent,late,leave',
            'attendance.*.remark' => 'nullable|string|max:255',
        ]);

        $date = $validated['date'];
        $classId = $validated['class_id'];
        $sectionId = $validated['section_id'];

        $teacher = Teacher::where('user_id', $user->id)->first();

        foreach ($validated['attendance'] as $studentId => $row) {
            Attendance::updateOrCreate(
                ['date' => $date, 'student_id' => $studentId],
                [
                    'class_id' => $classId,
                    'section_id' => $sectionId,
                    'teacher_id' => $teacher?->id,
                    'status' => $row['status'],
                    'remark' => $row['remark'] ?? null,
                ]
            );
        }

        return redirect()->back()->with('success', 'Attendance saved for '.$date);
    }
}


