<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\FeesCollection;
use App\Services\PaymentGateway\PaymentGatewayManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ParentPaymentController extends Controller
{
    public function __construct(protected PaymentGatewayManager $gatewayManager)
    {
    }

    /**
     * Start payment for a particular fees collection record.
     */
    public function start(FeesCollection $collection)
    {
        $user = Auth::user();
        if (!$user || !$user->isParent()) {
            abort(403, 'Unauthorized access');
        }

        // Ensure this fees record belongs to one of the parent's children
        if (!$collection->student || $collection->student->parent_id !== $user->id) {
            abort(403, 'You cannot pay for this record.');
        }

        $gateway = $this->gatewayManager->getActiveGateway();
        if (!$gateway) {
            return redirect()->route('parent.fees')
                ->with('error', 'No payment gateway is active. Please contact school admin.');
        }

        $orderData = $gateway->createOrder($collection);

        // For now just redirect back with stub data; later this can redirect to gateway URL or show a payment page.
        return redirect()->route('parent.fees')
            ->with('success', 'Payment initialized with ' . $orderData['gateway'] . ' (demo).');
    }

    /**
     * View/print receipt for a paid collection.
     */
    public function receipt(FeesCollection $collection)
    {
        $user = Auth::user();
        if (!$user || !$user->isParent()) {
            abort(403, 'Unauthorized access');
        }

        if (!$collection->student || $collection->student->parent_id !== $user->id) {
            abort(403, 'You cannot access this receipt.');
        }

        return view('parent.fees.receipt', [
            'collection' => $collection,
            'user' => $user,
        ]);
    }
}


