<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Homework;
use App\Models\HomeworkSubmission;
use App\Models\Student;
use App\Models\AdminNotification;
use App\Models\TeacherNotification;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;

class ParentHomeworkSubmissionController extends Controller
{
    public function store(Request $request, $homeworkId)
    {
        $user = Auth::user();
        if (!$user || !$user->isParent()) {
            abort(403, 'Unauthorized access');
        }

        $homework = Homework::findOrFail($homeworkId);
        
        // Verify the student belongs to this parent
        $student = Student::where('id', $request->student_id)
            ->where('parent_id', $user->id)
            ->firstOrFail();

        // Verify homework is for this student's class/section
        if ($homework->class_id != $student->class_id) {
            abort(403, 'This homework is not assigned to this student.');
        }

        if ($homework->section_id && $homework->section_id != $student->section_id) {
            abort(403, 'This homework is not assigned to this student.');
        }

        $validated = $request->validate([
            'student_id' => 'required|exists:students,id',
            'submitted_date' => 'required|date',
            'submission_notes' => 'nullable|string',
            'attachments' => 'nullable|array',
            'attachments.*' => 'file|max:10240', // 10MB max per file
        ]);

        // Determine if submission is late
        $submittedDate = Carbon::parse($validated['submitted_date']);
        $dueDate = Carbon::parse($homework->due_date);
        $status = $submittedDate->isAfter($dueDate) ? 'late' : 'submitted';

        // Handle file uploads
        $attachmentPaths = [];
        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $path = $file->store('homework/submissions', 'public');
                $attachmentPaths[] = $path;
            }
        }

        // Update or create submission
        $submission = HomeworkSubmission::updateOrCreate(
            [
                'homework_id' => $homework->id,
                'student_id' => $student->id,
            ],
            [
                'submitted_date' => $validated['submitted_date'],
                'submission_notes' => $validated['submission_notes'] ?? null,
                'attachments' => !empty($attachmentPaths) ? $attachmentPaths : null,
                'status' => $status,
            ]
        );

        // Send admin notification
        try {
            AdminNotification::create([
                'type' => 'homework_submission',
                'title' => 'Homework Submitted',
                'message' => "{$student->name} submitted homework: {$homework->title}" . ($status == 'late' ? ' (Late)' : ''),
                'icon' => 'ri-book-open-line',
                'icon_bg' => $status == 'late' ? 'bg-warning-600' : 'bg-success-600',
                'link' => route('admin.homework.show', $homework->id),
                'from_user_id' => $user->id,
                'from_role' => 'student',
                'data' => [
                    'homework_id' => $homework->id,
                    'submission_id' => $submission->id,
                    'student_id' => $student->id,
                    'student_name' => $student->name,
                ],
            ]);
            
            // Send teacher notification
            TeacherNotification::notifyHomeworkSubmission($submission, $student, $homework);
        } catch (\Exception $e) {
            \Log::error('Failed to create notification: ' . $e->getMessage());
        }

        return redirect()->route('parent.homework.show', $homework->id)
            ->with('success', 'Homework submitted successfully.');
    }
}
