<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Homework;
use App\Models\Student;
use App\Models\HomeworkSubmission;

class ParentHomeworkController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isParent()) {
            abort(403, 'Unauthorized access');
        }

        // Load children with class & section relations
        $user->load(['students.schoolClass', 'students.section']);
        $children = $user->students;

        $selectedStudentId = $request->input('student_id');
        
        // If no selection and only one child, auto-select
        if (!$selectedStudentId && $children->count() === 1) {
            $selectedStudentId = $children->first()->id;
        }

        $homework = collect();
        $selectedStudent = null;

        if ($selectedStudentId) {
            $selectedStudent = $children->firstWhere('id', $selectedStudentId);
            
            if ($selectedStudent) {
                // Get homework for the student's class and section
                $homework = Homework::with(['teacher', 'subject'])
                    ->where('class_id', $selectedStudent->class_id)
                    ->where(function($query) use ($selectedStudent) {
                        $query->whereNull('section_id')
                              ->orWhere('section_id', $selectedStudent->section_id);
                    })
                    ->where('is_published', true)
                    ->orderByDesc('assigned_date')
                    ->orderByDesc('created_at')
                    ->get();
                
                // Load submissions for this student
                $homework->load(['submissions' => function($query) use ($selectedStudent) {
                    $query->where('student_id', $selectedStudent->id);
                }]);
            }
        }

        return view('parent.homework.index', [
            'user' => $user,
            'children' => $children,
            'selectedStudent' => $selectedStudent,
            'selectedStudentId' => $selectedStudentId,
            'homework' => $homework,
        ]);
    }

    public function show(string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isParent()) {
            abort(403, 'Unauthorized access');
        }

        $user->load(['students.schoolClass', 'students.section']);
        $children = $user->students;
        $childIds = $children->pluck('id')->toArray();
        $classIds = $children->pluck('class_id')->filter()->unique()->toArray();
        $sectionIds = $children->pluck('section_id')->filter()->unique()->toArray();

        $homework = Homework::with(['teacher', 'schoolClass', 'section', 'subject', 'submissions.student'])
            ->where('is_published', true)
            ->whereIn('class_id', $classIds)
            ->where(function($query) use ($sectionIds) {
                $query->whereNull('section_id')
                      ->orWhereIn('section_id', $sectionIds);
            })
            ->findOrFail($id);

        // Get submission for each child if they have one
        $submissions = [];
        foreach ($children as $child) {
            $submission = HomeworkSubmission::where('homework_id', $homework->id)
                ->where('student_id', $child->id)
                ->first();
            if ($submission) {
                $submissions[$child->id] = $submission;
            }
        }

        return view('parent.homework.show', compact('homework', 'user', 'children', 'submissions'));
    }
}
