<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\FeesCollection;
use App\Models\Timetable;
use App\Models\TimetableOverride;
use App\Models\Attendance;
use App\Models\Note;
use App\Models\ExamResult;
use App\Models\ExamSchedule;
use App\Models\Room;
use Carbon\Carbon;

class ParentDashboardController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        if (!$user || !$user->isParent()) {
            abort(403, 'Unauthorized access');
        }

        $user->load(['students.schoolClass', 'students.section', 'students.academicYear']);

        $children = $user->students;
        $childCount = $children->count();
        $childIds = $children->pluck('id')->all();

        // Fees aggregates
        $collections = FeesCollection::with('student')
            ->whereHas('student', fn ($q) => $q->where('parent_id', $user->id))
            ->get();

        $totalAmount = $collections->sum('amount');
        $totalPaid = $collections->sum('paid_amount');
        $totalDue = max(0, $totalAmount - $totalPaid);
        $pendingCount = $collections->filter(fn ($c) => ($c->amount - $c->paid_amount) > 0)->count();

        // Notes for parent's children: for_all or targeted to their classes
        $classIds = $children->pluck('class_id')->filter()->unique()->values();
        $notes = Note::with('schoolClass')
            ->where(function ($q) use ($classIds) {
                $q->where('for_all', true)
                  ->orWhereIn('class_id', $classIds);
            })
            ->orderByDesc('created_at')
            ->limit(10)
            ->get();

        // Find top marks achievements
        $topMarksAchievements = collect();
        if ($childIds) {
            // Get all exam results for parent's children
            $allResults = ExamResult::with([
                    'student',
                    'examSchedule.examType',
                    'examSchedule.subject',
                    'examSchedule.schoolClass',
                    'examSchedule.section',
                ])
                ->whereIn('student_id', $childIds)
                ->where('is_absent', false)
                ->whereNotNull('marks_obtained')
                ->get();

            // Group by exam schedule and find top 3 in each
            $resultsByExam = $allResults->groupBy('exam_schedule_id');
            
            foreach ($resultsByExam as $examId => $examResults) {
                // Sort by marks descending and get top 3
                $topThree = $examResults->sortByDesc('marks_obtained')->take(3);
                
                // Check if any of parent's children are in top 3
                foreach ($topThree as $index => $result) {
                    if (in_array($result->student_id, $childIds)) {
                        $topMarksAchievements->push([
                            'student' => $result->student,
                            'examSchedule' => $result->examSchedule,
                            'result' => $result,
                            'rank' => $index + 1,
                            'marks' => $result->marks_obtained,
                            'maxMarks' => $result->examSchedule->max_marks ?? 100,
                            'percentage' => $result->examSchedule->max_marks > 0 
                                ? round(($result->marks_obtained / $result->examSchedule->max_marks) * 100, 2) 
                                : 0,
                        ]);
                        break; // Only show once per exam
                    }
                }
            }
        }

        // Recent exam results for the parent's children (for the table)
        $recentResults = ExamResult::with([
                'student.schoolClass',
                'student.section',
                'examSchedule.examType',
                'examSchedule.subject',
            ])
            ->whereIn('student_id', $childIds)
            ->orderByDesc('created_at')
            ->limit(5)
            ->get();

        // Shared date/time values
        $now = Carbon::now();
        $today = $now->toDateString();
        $dayKey = strtolower($now->format('l')); // matches stored day_of_week values
        $timeNow = $now->format('H:i:s');

        // Upcoming exams for the parent's children (next 30 days)
        $upcomingExams = ExamSchedule::with(['examType','subject','schoolClass','section'])
            ->whereDate('exam_date', '>=', $today)
            ->whereDate('exam_date', '<=', $now->copy()->addDays(30)->toDateString())
            ->whereIn('class_id', $children->pluck('class_id')->filter()->unique())
            ->orderBy('exam_date')
            ->orderBy('start_time')
            ->limit(10)
            ->get();
        // Build attendance stats and timetable info for each child
        $scheduleInfo = [];
        $attendanceStats = [];
        $todayTimetable = [];
        $tomorrowTimetable = [];
        
        $tomorrow = $now->copy()->addDay()->toDateString();
        $tomorrowDayKey = strtolower($now->copy()->addDay()->format('l'));

        foreach ($children as $student) {
            // Attendance stats for this month
            $monthStart = $now->copy()->startOfMonth();
            $monthEnd = $now->copy()->endOfMonth();
            
            $monthAttendance = Attendance::where('student_id', $student->id)
                ->whereBetween('date', [$monthStart, $monthEnd])
                ->get();
            
            $attendanceStats[$student->id] = [
                'present' => $monthAttendance->where('status', 'present')->count(),
                'absent' => $monthAttendance->where('status', 'absent')->count(),
                'late' => $monthAttendance->where('status', 'late')->count(),
                'leave' => $monthAttendance->where('status', 'leave')->count(),
                'total' => $monthAttendance->count(),
                'percentage' => $monthAttendance->count() > 0 
                    ? round(($monthAttendance->where('status', 'present')->count() / $monthAttendance->count()) * 100, 1) 
                    : 0,
            ];

            // Current class: prefer override for today, else base timetable
            $currentOverride = TimetableOverride::with(['subject','teacher'])
                ->where('class_id', $student->class_id)
                ->where('section_id', $student->section_id)
                ->whereDate('date', $today)
                ->where('start_time', '<=', $timeNow)
                ->where('end_time', '>', $timeNow)
                ->orderBy('start_time')
                ->first();

            $currentBase = null;
            if (!$currentOverride) {
                $currentBase = Timetable::with(['subject','teacher'])
                    ->where('class_id', $student->class_id)
                    ->where('section_id', $student->section_id)
                    ->where('day_of_week', $dayKey)
                    ->where('start_time', '<=', $timeNow)
                    ->where('end_time', '>', $timeNow)
                    ->orderBy('start_time')
                    ->first();
            }

            // Next class today: pick earliest between override and base after now
            $nextOverride = TimetableOverride::with(['subject','teacher'])
                ->where('class_id', $student->class_id)
                ->where('section_id', $student->section_id)
                ->whereDate('date', $today)
                ->where('start_time', '>', $timeNow)
                ->orderBy('start_time')
                ->first();

            $nextBase = Timetable::with(['subject','teacher'])
                ->where('class_id', $student->class_id)
                ->where('section_id', $student->section_id)
                ->where('day_of_week', $dayKey)
                ->where('start_time', '>', $timeNow)
                ->orderBy('start_time')
                ->first();

            $nextSlot = null;
            if ($nextOverride && $nextBase) {
                $nextSlot = $nextOverride->start_time <= $nextBase->start_time ? $nextOverride : $nextBase;
            } else {
                $nextSlot = $nextOverride ?? $nextBase;
            }

            $currentSlot = $currentOverride ?? $currentBase;

            $scheduleInfo[$student->id] = [
                'current' => $currentSlot ? [
                    'subject' => $currentSlot->subject->name ?? '-',
                    'teacher' => $currentSlot->teacher->name ?? '-',
                    'start' => $currentSlot->start_time,
                    'end' => $currentSlot->end_time,
                ] : null,
                'next' => $nextSlot ? [
                    'subject' => $nextSlot->subject->name ?? '-',
                    'teacher' => $nextSlot->teacher->name ?? '-',
                    'start' => $nextSlot->start_time,
                    'end' => $nextSlot->end_time,
                ] : null,
            ];

            // Today's timetable
            $todayOverrides = TimetableOverride::with(['subject','teacher','roomRelation'])
                ->where('class_id', $student->class_id)
                ->where('section_id', $student->section_id)
                ->whereDate('date', $today)
                ->orderBy('start_time')
                ->get();

            $todayBase = Timetable::with(['subject','teacher'])
                ->where('class_id', $student->class_id)
                ->where('section_id', $student->section_id)
                ->where('day_of_week', $dayKey)
                ->orderBy('start_time')
                ->get();

            // Merge today's timetable (overrides take priority)
            $todaySlots = collect();
            $overrideTimes = $todayOverrides->map(function($ov) {
                return $ov->start_time . '-' . $ov->end_time;
            })->toArray();
            
            foreach ($todayBase as $base) {
                $timeKey = $base->start_time . '-' . $base->end_time;
                if (!in_array($timeKey, $overrideTimes)) {
                    $todaySlots->push($base);
                }
            }
            // Add all overrides
            foreach ($todayOverrides as $override) {
                $todaySlots->push($override);
            }
            $todayTimetable[$student->id] = $todaySlots->sortBy('start_time')->values();

            // Tomorrow's timetable
            $tomorrowOverrides = TimetableOverride::with(['subject','teacher','roomRelation'])
                ->where('class_id', $student->class_id)
                ->where('section_id', $student->section_id)
                ->whereDate('date', $tomorrow)
                ->orderBy('start_time')
                ->get();

            $tomorrowBase = Timetable::with(['subject','teacher'])
                ->where('class_id', $student->class_id)
                ->where('section_id', $student->section_id)
                ->where('day_of_week', $tomorrowDayKey)
                ->orderBy('start_time')
                ->get();

            // Merge tomorrow's timetable
            $tomorrowSlots = collect();
            $tomorrowOverrideTimes = $tomorrowOverrides->map(function($ov) {
                return $ov->start_time . '-' . $ov->end_time;
            })->toArray();
            
            foreach ($tomorrowBase as $base) {
                $timeKey = $base->start_time . '-' . $base->end_time;
                if (!in_array($timeKey, $tomorrowOverrideTimes)) {
                    $tomorrowSlots->push($base);
                }
            }
            foreach ($tomorrowOverrides as $override) {
                $tomorrowSlots->push($override);
            }
            $tomorrowTimetable[$student->id] = $tomorrowSlots->sortBy('start_time')->values();
        }

        return view('parent.dashboard', [
            'user' => $user,
            'children' => $children,
            'childCount' => $childCount,
            'totalAmount' => $totalAmount,
            'totalPaid' => $totalPaid,
            'totalDue' => $totalDue,
            'pendingCount' => $pendingCount,
            'scheduleInfo' => $scheduleInfo,
            'attendanceStats' => $attendanceStats,
            'todayTimetable' => $todayTimetable,
            'tomorrowTimetable' => $tomorrowTimetable,
            'notes' => $notes,
            'recentResults' => $recentResults,
            'upcomingExams' => $upcomingExams,
            'topMarksAchievements' => $topMarksAchievements,
        ]);
    }

    public function fees()
    {
        $user = Auth::user();
        if (!$user || !$user->isParent()) {
            abort(403, 'Unauthorized access');
        }

        // All fees records for this parent's children
        $collections = FeesCollection::with(['student.schoolClass','feesType'])
            ->whereHas('student', function ($q) use ($user) {
                $q->where('parent_id', $user->id);
            })
            ->orderByDesc('payment_date')
            ->get();

        return view('parent.fees.index', ['user' => $user, 'collections' => $collections]);
    }

    /**
     * Show timetable for one or more children of the logged-in parent.
     */
    public function timetable(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isParent()) {
            abort(403, 'Unauthorized access');
        }

        // Load children with class & section relations
        $user->load(['students.schoolClass', 'students.section']);
        $students = $user->students;

        $selectedIds = $request->input('student_ids', []);
        if (!is_array($selectedIds)) {
            $selectedIds = [$selectedIds];
        }

        // If nothing selected and only one child, auto-select that child
        if (empty($selectedIds) && $students->count() === 1) {
            $selectedIds = [$students->first()->id];
        }

        $timetablesByStudent = [];

        foreach ($students as $student) {
            if (!in_array($student->id, $selectedIds)) {
                continue;
            }

            // Weekly base timetable (by class & section)
            $slots = Timetable::with(['schoolClass', 'section', 'subject', 'teacher'])
                ->where('class_id', $student->class_id)
                ->where('section_id', $student->section_id)
                ->orderBy('day_of_week')
                ->orderBy('start_time')
                ->get();

            // Date-wise overrides for this class/section
            $overrides = TimetableOverride::with(['schoolClass', 'section', 'subject', 'teacher'])
                ->where('class_id', $student->class_id)
                ->where('section_id', $student->section_id)
                ->orderBy('date')
                ->orderBy('start_time')
                ->get();

            $timetablesByStudent[$student->id] = [
                'student' => $student,
                'slots' => $slots,
                'overrides' => $overrides,
            ];
        }

        return view('parent.timetable.index', [
            'user' => $user,
            'students' => $students,
            'selectedIds' => $selectedIds,
            'timetablesByStudent' => $timetablesByStudent,
        ]);
    }

    /**
     * Show detailed profile for a child (admission + attendance snapshot).
     */
    public function student(\App\Models\Student $student)
    {
        $user = Auth::user();
        if (!$user || !$user->isParent()) {
            abort(403, 'Unauthorized access');
        }

        if ($student->parent_id !== $user->id) {
            abort(403, 'You cannot view this student.');
        }

        $student->load(['academicYear', 'schoolClass', 'section', 'batch']);

        $attendance = Attendance::with('teacher')
            ->where('student_id', $student->id)
            ->orderByDesc('date')
            ->limit(15)
            ->get();

        $attendanceStats = [
            'present' => $attendance->where('status', 'present')->count(),
            'absent' => $attendance->where('status', 'absent')->count(),
            'late' => $attendance->where('status', 'late')->count(),
        ];

        // Exam results for this student
        $examResults = ExamResult::with([
                'examSchedule.examType',
                'examSchedule.subject',
                'examSchedule.schoolClass',
                'examSchedule.section',
            ])
            ->where('student_id', $student->id)
            ->orderByDesc('created_at')
            ->limit(20)
            ->get();

        return view('parent.students.show', [
            'user' => $user,
            'student' => $student,
            'attendance' => $attendance,
            'attendanceStats' => $attendanceStats,
            'examResults' => $examResults,
        ]);
    }
}


