<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Attendance;
use App\Models\Student;
use Carbon\Carbon;

class ParentAttendanceController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isParent()) {
            abort(403, 'Unauthorized access');
        }

        // Load children with class & section relations
        $user->load(['students.schoolClass', 'students.section']);
        $children = $user->students;

        $selectedStudentId = $request->input('student_id');
        
        // If no selection and only one child, auto-select
        if (!$selectedStudentId && $children->count() === 1) {
            $selectedStudentId = $children->first()->id;
        }

        // Get date range (default: current month)
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->toDateString());
        $endDate = $request->input('end_date', Carbon::now()->endOfMonth()->toDateString());

        $attendanceRecords = collect();
        $attendanceStats = null;
        $selectedStudent = null;

        if ($selectedStudentId) {
            $selectedStudent = $children->firstWhere('id', $selectedStudentId);
            
            if ($selectedStudent) {
                // Get attendance records for selected date range
                $attendanceRecords = Attendance::with(['teacher', 'schoolClass', 'section'])
                    ->where('student_id', $selectedStudentId)
                    ->whereBetween('date', [$startDate, $endDate])
                    ->orderByDesc('date')
                    ->get();

                // Calculate stats
                $total = $attendanceRecords->count();
                $present = $attendanceRecords->where('status', 'present')->count();
                $absent = $attendanceRecords->where('status', 'absent')->count();
                $late = $attendanceRecords->where('status', 'late')->count();
                $leave = $attendanceRecords->where('status', 'leave')->count();
                $percentage = $total > 0 ? round(($present / $total) * 100, 1) : 0;

                $attendanceStats = [
                    'total' => $total,
                    'present' => $present,
                    'absent' => $absent,
                    'late' => $late,
                    'leave' => $leave,
                    'percentage' => $percentage,
                ];
            }
        }

        return view('parent.attendance.index', [
            'user' => $user,
            'children' => $children,
            'selectedStudent' => $selectedStudent,
            'selectedStudentId' => $selectedStudentId,
            'attendanceRecords' => $attendanceRecords,
            'attendanceStats' => $attendanceStats,
            'startDate' => $startDate,
            'endDate' => $endDate,
        ]);
    }
}
