<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Homework;
use App\Models\HomeworkSubmission;
use App\Models\Student;
use App\Models\Attendance;
use App\Models\FeesCollection;
use App\Models\FeesAssignment;
use App\Models\Setting;
use App\Models\ExamSchedule;
use App\Models\ExamResult;
use App\Models\Timetable;
use App\Models\LeaveRequest;
use App\Models\LeaveType;
use App\Models\Note;
use App\Models\Notification;
use App\Models\DeviceToken;
use App\Models\AdminNotification;

class ParentController extends Controller
{
    /**
     * Get parent dashboard data
     */
    public function dashboard(Request $request)
    {
        $user = $request->user();
        $children = $user->students()->with(['schoolClass', 'section'])->get();

        return response()->json([
            'success' => true,
            'data' => [
                'children' => $children,
            ]
        ], 200);
    }

    /**
     * Get parent's children
     */
    public function children(Request $request)
    {
        $user = $request->user();
        $children = $user->students()->with(['schoolClass', 'section', 'batch'])->get();

        // Add attendance percentage for each child
        $children->each(function ($child) {
            $totalDays = Attendance::where('student_id', $child->id)->count();
            $presentDays = Attendance::where('student_id', $child->id)
                ->where('status', 'present')
                ->count();
            
            if ($totalDays > 0) {
                $child->attendance_percentage = round(($presentDays / $totalDays) * 100, 1);
            } else {
                $child->attendance_percentage = 0;
            }
        });

        return response()->json([
            'success' => true,
            'data' => $children
        ], 200);
    }

    /**
     * Get homework for parent's children with submission status
     */
    public function homework(Request $request)
    {
        $user = $request->user();
        $children = $user->students;
        
        if ($children->isEmpty()) {
            return response()->json([
                'success' => true,
                'data' => [
                    'pending' => [],
                    'submitted' => [],
                    'overdue' => [],
                ]
            ], 200);
        }

        $childIds = $children->pluck('id')->toArray();
        $classIds = $children->pluck('class_id')->filter()->unique()->toArray();
        $sectionIds = $children->pluck('section_id')->filter()->unique()->toArray();

        // Get homework for the class/section
        $homeworkList = Homework::with(['teacher', 'subject'])
            ->whereIn('class_id', $classIds)
            ->whereIn('section_id', $sectionIds)
            ->where('is_published', true)
            ->orderByDesc('assigned_date')
            ->get();

        $pending = [];
        $submitted = [];
        $overdue = [];
        $now = now();

        foreach ($homeworkList as $homework) {
            // Get submissions for all children for this homework
            $submissions = HomeworkSubmission::whereIn('student_id', $childIds)
                ->where('homework_id', $homework->id)
                ->with('student')
                ->get();

            $homeworkData = [
                'id' => $homework->id,
                'title' => $homework->title,
                'description' => $homework->description,
                'subject' => $homework->subject->name ?? 'N/A',
                'teacher' => $homework->teacher->name ?? 'N/A',
                'assigned_date' => $homework->assigned_date->format('d M Y'),
                'due_date' => $homework->due_date->format('d M Y'),
                'status' => $homework->status,
                'notes' => $homework->notes,
                'attachments' => $homework->attachments ?? [],
                'submissions' => $submissions->map(function ($submission) {
                    return [
                        'student_id' => $submission->student_id,
                        'student_name' => $submission->student->name,
                        'submitted_date' => $submission->submitted_date ? $submission->submitted_date->format('d M Y') : null,
                        'status' => $submission->status,
                        'grade' => $submission->grade,
                        'teacher_notes' => $submission->teacher_notes,
                        'attachments' => $submission->attachments ?? [],
                    ];
                })->toArray(),
            ];

            // Determine category based on submission status for all children
            $allChildrenSubmitted = true;
            $anyChildSubmitted = false;
            $hasAnySubmission = $submissions->isNotEmpty();

            foreach ($childIds as $childId) {
                $submission = $submissions->where('student_id', $childId)->first();
                if ($submission && $submission->status === 'submitted') {
                    $anyChildSubmitted = true;
                } else {
                    $allChildrenSubmitted = false;
                }
            }

            // Categorization logic:
            // 1. Submitted: ALL children have submitted (and at least one submission exists)
            // 2. Overdue: Past due date AND not all children submitted
            // 3. Pending: Not past due OR no submissions yet
            
            $isPastDue = $homework->due_date->isPast();
            
            if ($allChildrenSubmitted && $anyChildSubmitted && $hasAnySubmission) {
                // All children submitted
                $submitted[] = $homeworkData;
            } elseif ($isPastDue && !$allChildrenSubmitted) {
                // Past due and not all submitted
                $overdue[] = $homeworkData;
            } else {
                // Not past due or no submissions yet
                $pending[] = $homeworkData;
            }
        }

        return response()->json([
            'success' => true,
            'data' => [
                'pending' => $pending,
                'submitted' => $submitted,
                'overdue' => $overdue,
            ]
        ], 200);
    }

    /**
     * Get attendance for parent's children
     */
    public function attendance(Request $request)
    {
        $user = $request->user();
        $children = $user->students;
        $childIds = $children->pluck('id')->toArray();

        $attendance = Attendance::whereIn('student_id', $childIds)
            ->with(['student', 'teacher'])
            ->orderByDesc('date')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $attendance
        ], 200);
    }

    /**
     * Get fees for parent's children
     */
    public function fees(Request $request)
    {
        $user = $request->user();
        $children = $user->students;
        
        // If student_id is provided, filter for that student only
        $studentId = $request->query('student_id');
        if ($studentId && $studentId !== 'all') {
            $childIds = [$studentId];
        } else {
            $childIds = $children->pluck('id')->toArray();
        }

        // Get all fee collections
        $collections = FeesCollection::whereIn('student_id', $childIds)
            ->with(['student', 'feesType', 'payments'])
            ->orderByDesc('payment_date')
            ->get();

        // Separate into pending (partially paid) and history (fully paid)
        $pending = [];
        $history = [];

        foreach ($collections as $collection) {
            // Calculate total paid from payments table
            $totalPaid = $collection->payments->sum('amount');
            $dueAmount = $collection->amount - $totalPaid;

            // Get latest payment info for history
            $latestPayment = $collection->payments->first();
            
            $feeData = [
                'id' => $collection->id,
                'student_id' => $collection->student_id,
                'student_name' => $collection->student->name ?? 'N/A',
                'fee_type' => $collection->feesType->name ?? 'N/A',
                'amount' => (float) $collection->amount,
                'paid_amount' => (float) $totalPaid,
                'due_amount' => (float) max(0, $dueAmount),
                'discount_amount' => (float) ($collection->discount_amount ?? 0),
                'due_date' => $collection->payment_date ? $collection->payment_date->format('d M Y') : null,
                'payment_date' => $latestPayment ? $latestPayment->payment_date->format('d M Y') : ($collection->payment_date ? $collection->payment_date->format('d M Y') : null),
                'payment_method' => $latestPayment ? $latestPayment->payment_method : ($collection->payment_method ?? null),
                'reference' => $latestPayment ? $latestPayment->reference : ($collection->reference ?? null),
                'frequency' => $collection->feesType->recurring_frequency ?? null,
            ];

            if ($dueAmount > 0) {
                // Partially paid or unpaid - goes to pending
                $feeData['status'] = 'pending';
                $pending[] = $feeData;
            } else {
                // Fully paid - goes to history
                $feeData['status'] = 'paid';
                $history[] = $feeData;
            }
        }

        // Get upcoming fee assignments (not yet invoiced)
        $assignments = FeesAssignment::whereIn('student_id', $childIds)
            ->with(['student', 'feesType'])
            ->where('is_active', true)
            ->get();

        $upcoming = [];
        foreach ($assignments as $assignment) {
            // Check if this assignment has any collections
            $hasCollections = FeesCollection::where('student_id', $assignment->student_id)
                ->where('fees_type_id', $assignment->fees_type_id)
                ->exists();

            // Only show as upcoming if no collections exist yet or if it's a future date
            if (!$hasCollections || ($assignment->start_date && $assignment->start_date->isFuture())) {
                $upcoming[] = [
                    'id' => $assignment->id,
                    'student_id' => $assignment->student_id,
                    'student_name' => $assignment->student->name,
                    'fee_type' => $assignment->feesType->name ?? 'N/A',
                    'amount' => $assignment->amount,
                    'paid_amount' => 0,
                    'due_amount' => $assignment->amount,
                    'due_date' => $assignment->start_date ? $assignment->start_date->format('d M Y') : null,
                    'frequency' => $assignment->frequency,
                    'status' => 'upcoming',
                ];
            }
        }

        // Prepare children list for dropdown
        $childrenList = $user->students->map(function($child) {
            return [
                'id' => $child->id,
                'name' => $child->name,
            ];
        });

        return response()->json([
            'success' => true,
            'data' => [
                'children' => $childrenList,
                'pending' => $pending,
                'history' => $history,
                'upcoming' => $upcoming,
                'summary' => [
                    'total_pending' => array_sum(array_column($pending, 'due_amount')),
                    'total_paid' => array_sum(array_column($history, 'paid_amount')),
                    'total_upcoming' => array_sum(array_column($upcoming, 'due_amount')),
                ],
            ]
        ], 200);
    }

    /**
     * Get payment gateway settings
     */
    public function paymentSettings(Request $request)
    {
        $settings = Setting::first();

        if (!$settings) {
            return response()->json([
                'success' => false,
                'message' => 'Payment settings not configured'
            ], 404);
        }

        $paymentGateways = [];

        // Razorpay
        if ($settings->razorpay_active) {
            $paymentGateways[] = [
                'name' => 'razorpay',
                'label' => 'Razorpay',
                'key_id' => $settings->razorpay_key_id,
                'active' => true,
            ];
        }

        // Cashfree
        if ($settings->cashfree_active) {
            $paymentGateways[] = [
                'name' => 'cashfree',
                'label' => 'Cashfree',
                'app_id' => $settings->cashfree_app_id,
                'mode' => $settings->cashfree_mode,
                'active' => true,
            ];
        }

        // PhonePe
        if ($settings->phonepe_active) {
            $paymentGateways[] = [
                'name' => 'phonepe',
                'label' => 'PhonePe',
                'merchant_id' => $settings->phonepe_merchant_id,
                'mode' => $settings->phonepe_mode,
                'active' => true,
            ];
        }

        return response()->json([
            'success' => true,
            'data' => [
                'gateways' => $paymentGateways,
                'default_gateway' => $paymentGateways[0] ?? null,
            ]
        ], 200);
    }

    /**
     * Initiate payment for a fee
     */
    public function initiatePayment(Request $request)
    {
        $validated = $request->validate([
            'fee_id' => 'required|integer|exists:fees_collections,id',
            'amount' => 'required|numeric|min:1',
            'gateway' => 'required|string|in:razorpay,cashfree,phonepe',
        ]);

        $user = $request->user();
        
        // Get the fee collection
        $fee = FeesCollection::with('student')->findOrFail($validated['fee_id']);

        // Verify the fee belongs to one of the user's children
        $childIds = $user->students->pluck('id')->toArray();
        if (!in_array($fee->student_id, $childIds)) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized access to this fee'
            ], 403);
        }

        // Get payment settings
        $settings = Setting::first();
        if (!$settings) {
            return response()->json([
                'success' => false,
                'message' => 'Payment settings not configured'
            ], 500);
        }

        // Generate order ID
        $orderId = 'FEE_' . $fee->id . '_' . time();

        // Prepare payment data based on gateway
        $paymentData = [
            'order_id' => $orderId,
            'amount' => $validated['amount'],
            'currency' => 'INR',
            'description' => $fee->feesType->name . ' - ' . $fee->student->name,
            'student_name' => $fee->student->name,
            'parent_name' => $user->name,
            'parent_email' => $user->email,
            'parent_phone' => $user->phone ?? '',
        ];

        // Add gateway-specific data and create payment session
        if ($validated['gateway'] === 'razorpay') {
            $paymentData['key_id'] = $settings->razorpay_key_id;
            $paymentData['gateway'] = 'razorpay';
            // Razorpay requires native SDK integration
        } elseif ($validated['gateway'] === 'cashfree') {
            // Create Cashfree Order for native SDK payment
            try {
                $baseUrl = in_array($settings->cashfree_mode, ['test', 'sandbox']) 
                    ? 'https://sandbox.cashfree.com/pg/orders'
                    : 'https://api.cashfree.com/pg/orders';

                $orderPayload = [
                    'order_id' => $orderId,
                    'order_amount' => (float) $validated['amount'],
                    'order_currency' => 'INR',
                    'customer_details' => [
                        'customer_id' => 'PARENT_' . $user->id,
                        'customer_name' => $user->name,
                        'customer_email' => $user->email,
                        'customer_phone' => $user->phone ?? '9999999999',
                    ],
                    'order_note' => ($fee->feesType->name ?? 'Fee') . ' Payment',
                ];

                $ch = curl_init($baseUrl);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($orderPayload));
                curl_setopt($ch, CURLOPT_HTTPHEADER, [
                    'Content-Type: application/json',
                    'x-api-version: 2023-08-01',
                    'x-client-id: ' . $settings->cashfree_app_id,
                    'x-client-secret: ' . $settings->cashfree_secret_key,
                ]);
                
                $response = curl_exec($ch);
                $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                curl_close($ch);

                $responseData = json_decode($response, true);

                if ($httpCode === 200 && isset($responseData['payment_session_id'])) {
                    $paymentData['gateway'] = 'cashfree';
                    $paymentData['payment_session_id'] = $responseData['payment_session_id'];
                    $paymentData['cf_order_id'] = $responseData['cf_order_id'];
                    $paymentData['environment'] = in_array($settings->cashfree_mode, ['test', 'sandbox']) ? 'SANDBOX' : 'PRODUCTION';
                    $paymentData['app_id'] = $settings->cashfree_app_id;
                } else {
                    \Log::error('Cashfree order creation failed', [
                        'http_code' => $httpCode,
                        'response' => $responseData
                    ]);
                    return response()->json([
                        'success' => false,
                        'message' => 'Failed to create payment order: ' . ($responseData['message'] ?? 'Unknown error')
                    ], 500);
                }
            } catch (\Exception $e) {
                \Log::error('Cashfree API error: ' . $e->getMessage());
                return response()->json([
                    'success' => false,
                    'message' => 'Payment gateway error: ' . $e->getMessage()
                ], 500);
            }
        } elseif ($validated['gateway'] === 'phonepe') {
            $paymentData['merchant_id'] = $settings->phonepe_merchant_id;
            $paymentData['mode'] = $settings->phonepe_mode;
            $paymentData['gateway'] = 'phonepe';
        }

        return response()->json([
            'success' => true,
            'data' => $paymentData
        ], 200);
    }

    /**
     * Verify payment and update fee status
     */
    public function verifyPayment(Request $request)
    {
        $validated = $request->validate([
            'order_id' => 'required|string',
            'gateway' => 'required|string|in:razorpay,cashfree,phonepe',
        ]);

        $user = $request->user();
        $orderId = $validated['order_id'];
        
        // Extract fee_id from order_id (format: FEE_{fee_id}_{timestamp})
        $parts = explode('_', $orderId);
        if (count($parts) < 2 || $parts[0] !== 'FEE') {
            return response()->json([
                'success' => false,
                'message' => 'Invalid order ID format'
            ], 400);
        }
        
        $feeId = $parts[1];
        $fee = FeesCollection::with(['student', 'feesType'])->find($feeId);
        
        if (!$fee) {
            return response()->json([
                'success' => false,
                'message' => 'Fee not found'
            ], 404);
        }

        // Verify the fee belongs to one of the user's children
        $childIds = $user->students->pluck('id')->toArray();
        if (!in_array($fee->student_id, $childIds)) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized access to this fee'
            ], 403);
        }

        // Get payment settings
        $settings = Setting::first();

        if ($validated['gateway'] === 'cashfree') {
            try {
                $isSandbox = in_array($settings->cashfree_mode, ['test', 'sandbox']);
                
                // First, check order status
                $orderUrl = $isSandbox
                    ? 'https://sandbox.cashfree.com/pg/orders/' . $orderId
                    : 'https://api.cashfree.com/pg/orders/' . $orderId;

                $ch = curl_init($orderUrl);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_HTTPHEADER, [
                    'Content-Type: application/json',
                    'x-api-version: 2023-08-01',
                    'x-client-id: ' . $settings->cashfree_app_id,
                    'x-client-secret: ' . $settings->cashfree_secret_key,
                ]);

                $response = curl_exec($ch);
                $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                curl_close($ch);

                $orderData = json_decode($response, true);

                \Log::info('Cashfree order verification', [
                    'order_id' => $orderId,
                    'http_code' => $httpCode,
                    'response' => $orderData
                ]);

                $isPaid = false;
                $paidAmount = $fee->due_amount ?? $fee->amount;
                $transactionId = $orderId;

                // Check if order status is PAID
                if ($httpCode === 200 && isset($orderData['order_status']) && $orderData['order_status'] === 'PAID') {
                    $isPaid = true;
                    $paidAmount = $orderData['order_amount'] ?? $paidAmount;
                    $transactionId = $orderData['cf_order_id'] ?? $orderId;
                }
                
                // If order not found or not PAID, check payments endpoint
                if (!$isPaid && $httpCode === 200) {
                    $paymentsUrl = $isSandbox
                        ? 'https://sandbox.cashfree.com/pg/orders/' . $orderId . '/payments'
                        : 'https://api.cashfree.com/pg/orders/' . $orderId . '/payments';

                    $ch = curl_init($paymentsUrl);
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($ch, CURLOPT_HTTPHEADER, [
                        'Content-Type: application/json',
                        'x-api-version: 2023-08-01',
                        'x-client-id: ' . $settings->cashfree_app_id,
                        'x-client-secret: ' . $settings->cashfree_secret_key,
                    ]);

                    $paymentsResponse = curl_exec($ch);
                    $paymentsHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                    curl_close($ch);

                    $paymentsData = json_decode($paymentsResponse, true);
                    
                    \Log::info('Cashfree payments check', [
                        'order_id' => $orderId,
                        'http_code' => $paymentsHttpCode,
                        'payments' => $paymentsData
                    ]);

                    // Check if any payment was successful
                    if ($paymentsHttpCode === 200 && is_array($paymentsData)) {
                        foreach ($paymentsData as $payment) {
                            if (isset($payment['payment_status']) && $payment['payment_status'] === 'SUCCESS') {
                                $isPaid = true;
                                $paidAmount = $payment['payment_amount'] ?? $paidAmount;
                                $transactionId = $payment['cf_payment_id'] ?? $orderId;
                                break;
                            }
                        }
                    }
                }

                // SDK callback success means payment was made - trust it if fee is still pending
                // This handles cases where Cashfree API has delays
                $trustSdkCallback = $request->input('trust_sdk', false);
                if (!$isPaid && $trustSdkCallback && $fee->status === 'pending') {
                    $isPaid = true;
                    \Log::info('Trusting SDK callback for payment', ['order_id' => $orderId]);
                }

                // Check if fee is already paid
                if ($fee->status === 'paid') {
                    return response()->json([
                        'success' => true,
                        'message' => 'Payment already verified',
                        'data' => [
                            'status' => 'paid',
                            'amount_paid' => $fee->paid_amount,
                        ]
                    ], 200);
                }

                if ($isPaid) {
                    // Payment successful - update fee status
                    $fee->status = 'paid';
                    $fee->paid_amount = ($fee->paid_amount ?? 0) + $paidAmount;
                    $fee->payment_date = now();
                    $fee->payment_method = 'cashfree';
                    $fee->save();

                    // Payment record creation is optional - skip if model doesn't exist or has issues
                    try {
                        if (class_exists('\App\Models\Payment')) {
                            \App\Models\Payment::create([
                                'fees_collection_id' => $fee->id,
                                'student_id' => $fee->student_id,
                                'amount' => $paidAmount,
                                'payment_method' => 'cashfree',
                                'status' => 'completed',
                                'payment_date' => now(),
                            ]);
                        }
                    } catch (\Exception $e) {
                        // Payment model may not exist or have issues, log and continue
                        \Log::info('Payment record creation skipped: ' . $e->getMessage());
                    }

                    return response()->json([
                        'success' => true,
                        'message' => 'Payment verified and fee updated successfully',
                        'data' => [
                            'status' => 'paid',
                            'amount_paid' => $paidAmount,
                        ]
                    ], 200);
                } else {
                    return response()->json([
                        'success' => false,
                        'message' => 'Payment not completed. Status: ' . ($orderData['order_status'] ?? 'unknown'),
                        'data' => [
                            'status' => $orderData['order_status'] ?? 'unknown',
                            'order_id' => $orderId
                        ]
                    ], 400);
                }
            } catch (\Exception $e) {
                \Log::error('Payment verification error: ' . $e->getMessage());
                return response()->json([
                    'success' => false,
                    'message' => 'Payment verification failed: ' . $e->getMessage()
                ], 500);
            }
        }

        return response()->json([
            'success' => false,
            'message' => 'Gateway not supported for verification'
        ], 400);
    }

    /**
     * Get exam schedules and results for parent's children
     */
    public function exams(Request $request)
    {
        $user = $request->user();
        $children = $user->students;
        $childIds = $children->pluck('id')->toArray();

        if (empty($childIds)) {
            return response()->json([
                'success' => true,
                'data' => [
                    'upcoming' => [],
                    'completed' => [],
                ]
            ], 200);
        }

        // Get class and section info for the children
        $firstChild = $children->first();
        $classId = $firstChild->class_id;
        $sectionId = $firstChild->section_id;

        // Get all exam schedules for the class/section
        $examSchedules = ExamSchedule::where('class_id', $classId)
            ->where('section_id', $sectionId)
            ->where('is_published', true)
            ->with(['examType', 'subject', 'room'])
            ->orderBy('exam_date', 'asc')
            ->orderBy('start_time', 'asc')
            ->get();

        $upcoming = [];
        $completed = [];
        $now = now();

        foreach ($examSchedules as $schedule) {
            $examDateTime = \Carbon\Carbon::parse($schedule->exam_date->format('Y-m-d') . ' ' . $schedule->end_time);
            
            // Get results for all children in this exam
            $results = ExamResult::where('exam_schedule_id', $schedule->id)
                ->whereIn('student_id', $childIds)
                ->with('student')
                ->get();

            $examData = [
                'id' => $schedule->id,
                'exam_type' => $schedule->examType->name ?? 'N/A',
                'subject' => $schedule->subject->name ?? 'N/A',
                'exam_date' => $schedule->exam_date->format('d M Y'),
                'start_time' => $schedule->start_time,
                'end_time' => $schedule->end_time,
                'room' => $schedule->room->name ?? $schedule->room ?? 'N/A',
                'max_marks' => $schedule->max_marks,
                'pass_mark' => $schedule->pass_mark,
                'note' => $schedule->note,
            ];

            // Check if exam is completed (past date)
            if ($examDateTime->isPast()) {
                // Add results if available
                $examData['results'] = $results->map(function ($result) use ($schedule) {
                    return [
                        'student_id' => $result->student_id,
                        'student_name' => $result->student->name,
                        'marks_obtained' => $result->marks_obtained,
                        'max_marks' => $schedule->max_marks,
                        'percentage' => round(($result->marks_obtained / $schedule->max_marks) * 100, 2),
                        'grade' => $result->grade,
                        'is_absent' => $result->is_absent,
                        'remark' => $result->remark,
                        'status' => $result->is_absent ? 'Absent' : ($result->marks_obtained >= $schedule->pass_mark ? 'Pass' : 'Fail'),
                    ];
                })->toArray();
                
                $completed[] = $examData;
            } else {
                // Upcoming exam
                $upcoming[] = $examData;
            }
        }

        return response()->json([
            'success' => true,
            'data' => [
                'upcoming' => $upcoming,
                'completed' => $completed,
            ]
        ], 200);
    }

    /**
     * Get timetable for parent's children
     */
    public function timetable(Request $request)
    {
        $user = $request->user();
        $children = $user->students;

        if ($children->isEmpty()) {
            return response()->json([
                'success' => true,
                'data' => []
            ], 200);
        }

        // Get class and section info for the children
        $firstChild = $children->first();
        $classId = $firstChild->class_id;
        $sectionId = $firstChild->section_id;

        // Get all timetable entries for the class/section
        $timetables = Timetable::where('class_id', $classId)
            ->where('section_id', $sectionId)
            ->with(['subject', 'teacher'])
            ->orderByRaw("CASE day_of_week 
                WHEN 'monday' THEN 1 
                WHEN 'tuesday' THEN 2 
                WHEN 'wednesday' THEN 3 
                WHEN 'thursday' THEN 4 
                WHEN 'friday' THEN 5 
                WHEN 'saturday' THEN 6 
                WHEN 'sunday' THEN 7 
                END")
            ->orderBy('start_time', 'asc')
            ->get();

        // Group by day of week
        $groupedTimetable = [];
        $days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];

        foreach ($days as $day) {
            $daySchedule = $timetables->where('day_of_week', $day)->map(function ($timetable) {
                return [
                    'id' => $timetable->id,
                    'subject' => $timetable->subject->name ?? 'N/A',
                    'teacher' => $timetable->teacher->name ?? 'N/A',
                    'start_time' => $timetable->start_time,
                    'end_time' => $timetable->end_time,
                    'room' => $timetable->room ?? 'N/A',
                ];
            })->values()->toArray();

            if (!empty($daySchedule)) {
                $groupedTimetable[$day] = $daySchedule;
            }
        }

        return response()->json([
            'success' => true,
            'data' => $groupedTimetable
        ], 200);
    }

    /**
     * Get leave requests for parent's children
     */
    public function leaveRequests(Request $request)
    {
        $user = $request->user();
        $children = $user->students;
        $childIds = $children->pluck('id')->toArray();

        if (empty($childIds)) {
            return response()->json([
                'success' => true,
                'data' => [
                    'pending' => [],
                    'approved' => [],
                    'rejected' => [],
                ]
            ], 200);
        }

        // Get all leave requests for the children
        $leaveRequests = LeaveRequest::whereIn('student_id', $childIds)
            ->with(['student', 'leaveType'])
            ->orderByDesc('created_at')
            ->get();

        $pending = [];
        $approved = [];
        $rejected = [];

        foreach ($leaveRequests as $leave) {
            $leaveData = [
                'id' => $leave->id,
                'student_id' => $leave->student_id,
                'student_name' => $leave->student->name,
                'leave_type' => $leave->leaveType->name ?? 'N/A',
                'leave_type_code' => $leave->leaveType->short_code ?? '',
                'from_date' => $leave->from_date->format('d M Y'),
                'to_date' => $leave->to_date->format('d M Y'),
                'days' => $leave->from_date->diffInDays($leave->to_date) + 1,
                'reason' => $leave->reason,
                'admin_remarks' => $leave->admin_remarks,
                'status' => $leave->status,
                'created_at' => $leave->created_at->format('d M Y'),
            ];

            if ($leave->status === 'pending') {
                $pending[] = $leaveData;
            } elseif ($leave->status === 'approved') {
                $approved[] = $leaveData;
            } elseif ($leave->status === 'rejected') {
                $rejected[] = $leaveData;
            }
        }

        return response()->json([
            'success' => true,
            'data' => [
                'pending' => $pending,
                'approved' => $approved,
                'rejected' => $rejected,
            ]
        ], 200);
    }

    /**
     * Get active leave types
     */
    public function leaveTypes(Request $request)
    {
        $leaveTypes = LeaveType::where('is_active', true)
            ->orderBy('name', 'asc')
            ->get()
            ->map(function ($type) {
                return [
                    'id' => $type->id,
                    'name' => $type->name,
                    'short_code' => $type->short_code,
                    'max_days' => $type->max_days,
                    'description' => $type->description,
                ];
            });

        return response()->json([
            'success' => true,
            'data' => $leaveTypes
        ], 200);
    }

    /**
     * Submit a new leave request
     */
    public function submitLeaveRequest(Request $request)
    {
        $validated = $request->validate([
            'student_id' => 'required|integer|exists:students,id',
            'leave_type_id' => 'required|integer|exists:leave_types,id',
            'from_date' => 'required|date',
            'to_date' => 'required|date|after_or_equal:from_date',
            'reason' => 'required|string|max:500',
        ]);

        $user = $request->user();
        
        // Verify the student belongs to this parent
        $childIds = $user->students->pluck('id')->toArray();
        if (!in_array($validated['student_id'], $childIds)) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized access to this student'
            ], 403);
        }

        // Create leave request
        $leaveRequest = LeaveRequest::create([
            'user_id' => $user->id,
            'role' => 'parent',
            'student_id' => $validated['student_id'],
            'leave_type_id' => $validated['leave_type_id'],
            'from_date' => $validated['from_date'],
            'to_date' => $validated['to_date'],
            'reason' => $validated['reason'],
            'status' => 'pending',
        ]);

        $leaveRequest->load(['student', 'leaveType']);

        // Send admin notification
        try {
            AdminNotification::notifyLeaveRequest($leaveRequest, $user, 'Parent');
        } catch (\Exception $e) {
            \Log::error('Failed to create admin notification: ' . $e->getMessage());
        }

        return response()->json([
            'success' => true,
            'message' => 'Leave request submitted successfully',
            'data' => [
                'id' => $leaveRequest->id,
                'student_name' => $leaveRequest->student->name,
                'leave_type' => $leaveRequest->leaveType->name,
                'from_date' => $leaveRequest->from_date->format('d M Y'),
                'to_date' => $leaveRequest->to_date->format('d M Y'),
                'status' => $leaveRequest->status,
            ]
        ], 201);
    }

    /**
     * Get site name from settings
     */
    public function siteName(Request $request)
    {
        $setting = Setting::first();
        $siteName = $setting ? ($setting->site_name ?? 'Tuition Management System') : 'Tuition Management System';

        return response()->json([
            'success' => true,
            'data' => [
                'site_name' => $siteName,
            ]
        ], 200);
    }

    /**
     * Get contact info from settings
     */
    public function contactInfo(Request $request)
    {
        $setting = Setting::first();
        
        $data = [
            'site_name' => $setting->site_name ?? 'Tuition Management System',
            'contact_phone' => $setting->contact_phone ?? null,
            'contact_email' => $setting->contact_email ?? null,
            'contact_whatsapp' => $setting->contact_whatsapp ?? null,
            'address' => $setting->address ?? null,
            'city' => $setting->city ?? null,
            'state' => $setting->state ?? null,
            'zip_code' => $setting->zip_code ?? null,
            'full_address' => $this->buildFullAddress($setting),
            'map_url' => $setting->map_url ?? null,
            'office_hours' => $setting->office_hours ?? 'Monday - Saturday: 9:00 AM - 6:00 PM',
            'website_url' => $setting->website_url ?? null,
            'facebook_url' => $setting->facebook_url ?? null,
            'instagram_url' => $setting->instagram_url ?? null,
        ];

        return response()->json([
            'success' => true,
            'data' => $data
        ], 200);
    }

    /**
     * Build full address from settings
     */
    private function buildFullAddress($setting)
    {
        if (!$setting) return null;
        
        $parts = [];
        if ($setting->address) $parts[] = $setting->address;
        if ($setting->city) $parts[] = $setting->city;
        if ($setting->state) $parts[] = $setting->state;
        if ($setting->zip_code) $parts[] = $setting->zip_code;
        
        return !empty($parts) ? implode(', ', $parts) : null;
    }

    /**
     * Get notifications for parent dashboard
     */
    public function notifications(Request $request)
    {
        $user = $request->user();
        $children = $user->students;
        
        $childIds = $children->pluck('id')->toArray();
        $classIds = $children->pluck('class_id')->filter()->unique()->toArray();
        $notifications = [];

        // Get deleted notification IDs for this user (admin notifications)
        $deletedNotificationIds = \DB::table('notification_reads')
            ->where('user_id', $user->id)
            ->whereNotNull('deleted_at')
            ->pluck('notification_id')
            ->toArray();

        // Get dismissed system notification keys for this user
        $dismissedSystemKeys = \DB::table('dismissed_notifications')
            ->where('user_id', $user->id)
            ->pluck('notification_key')
            ->toArray();

        // 1. Get persistent notifications from database (excluding deleted ones)
        $dbNotifications = Notification::active()
            ->forParent($user->id, $childIds, $classIds)
            ->whereNotIn('id', $deletedNotificationIds)
            ->orderByDesc('created_at')
            ->limit(20)
            ->get();

        foreach ($dbNotifications as $notification) {
            $notifications[] = [
                'id' => $notification->id,
                'type' => $notification->type,
                'title' => $notification->title,
                'message' => $notification->message,
                'icon' => $notification->icon,
                'color' => $notification->color,
                'date' => $notification->created_at->format('d M Y'),
                'datetime' => $notification->created_at->toIso8601String(),
                'is_read' => $notification->isReadBy($user->id),
                'action_url' => $notification->action_url,
                'source' => 'admin',
            ];
        }

        // 2. Payment Pending (auto-generated)
        if (!$children->isEmpty()) {
            $pendingFees = FeesCollection::whereIn('student_id', $childIds)
                ->where('status', 'pending')
                ->with('student')
                ->get();
            
            foreach ($pendingFees as $fee) {
                $notificationKey = 'fee_' . $fee->id;
                if (in_array($notificationKey, $dismissedSystemKeys)) continue;
                
                $notifications[] = [
                    'id' => $notificationKey,
                    'type' => 'fee',
                    'title' => 'Payment Pending',
                    'message' => "Payment of ₹{$fee->amount} is pending for {$fee->student->name}",
                    'icon' => 'payment',
                    'color' => 'orange',
                    'date' => $fee->created_at->format('d M Y'),
                    'datetime' => $fee->created_at->toIso8601String(),
                    'is_read' => false,
                    'action_url' => 'fees',
                    'source' => 'system',
                ];
            }

            // 3. Exam Assigned (auto-generated)
            $upcomingExams = ExamSchedule::whereIn('class_id', $classIds)
                ->where('exam_date', '>=', now())
                ->with('subject')
                ->orderBy('exam_date')
                ->limit(5)
                ->get();
            
            foreach ($upcomingExams as $exam) {
                $notificationKey = 'exam_' . $exam->id;
                if (in_array($notificationKey, $dismissedSystemKeys)) continue;
                
                $notifications[] = [
                    'id' => $notificationKey,
                    'type' => 'exam',
                    'title' => 'Upcoming Exam',
                    'message' => "{$exam->subject->name} exam on {$exam->exam_date->format('d M Y')}",
                    'icon' => 'school',
                    'color' => 'indigo',
                    'date' => $exam->exam_date->format('d M Y'),
                    'datetime' => $exam->exam_date->toIso8601String(),
                    'is_read' => false,
                    'action_url' => 'exams',
                    'source' => 'system',
                ];
            }

            // 4. Results Available (auto-generated)
            $recentResults = ExamResult::whereIn('student_id', $childIds)
                ->whereHas('examSchedule', function($q) {
                    $q->where('exam_date', '<=', now());
                })
                ->with(['student', 'examSchedule.subject'])
                ->orderByDesc('created_at')
                ->limit(5)
                ->get();
            
            foreach ($recentResults as $result) {
                $notificationKey = 'result_' . $result->id;
                if (in_array($notificationKey, $dismissedSystemKeys)) continue;
                
                $notifications[] = [
                    'id' => $notificationKey,
                    'type' => 'result',
                    'title' => 'Result Available',
                    'message' => "{$result->examSchedule->subject->name} result published for {$result->student->name}",
                    'icon' => 'assessment',
                    'color' => 'green',
                    'date' => $result->created_at->format('d M Y'),
                    'datetime' => $result->created_at->toIso8601String(),
                    'is_read' => false,
                    'action_url' => 'exams',
                    'source' => 'system',
                ];
            }

            // 5. Admin Notes (auto-generated from notes)
            $notes = Note::where(function($query) use ($classIds) {
                    $query->where('for_all', true)
                          ->orWhereIn('class_id', $classIds);
                })
                ->orderByDesc('created_at')
                ->limit(5)
                ->get();
            
            foreach ($notes as $note) {
                $notificationKey = 'note_' . $note->id;
                if (in_array($notificationKey, $dismissedSystemKeys)) continue;
                
                $notifications[] = [
                    'id' => $notificationKey,
                    'type' => 'announcement',
                    'title' => $note->title,
                    'message' => $note->body ?? $note->title,
                    'icon' => 'campaign',
                    'color' => 'purple',
                    'date' => $note->created_at->format('d M Y'),
                    'datetime' => $note->created_at->toIso8601String(),
                    'is_read' => false,
                    'action_url' => null,
                    'source' => 'system',
                ];
            }
        }

        // Sort by datetime (most recent first)
        usort($notifications, function($a, $b) {
            return strtotime($b['datetime']) - strtotime($a['datetime']);
        });

        return response()->json([
            'success' => true,
            'data' => $notifications
        ], 200);
    }

    /**
     * Get unread notification count
     */
    public function notificationCount(Request $request)
    {
        $user = $request->user();
        $children = $user->students;
        
        $childIds = $children->pluck('id')->toArray();
        $classIds = $children->pluck('class_id')->filter()->unique()->toArray();

        // Get deleted admin notification IDs
        $deletedNotificationIds = \DB::table('notification_reads')
            ->where('user_id', $user->id)
            ->whereNotNull('deleted_at')
            ->pluck('notification_id')
            ->toArray();

        // Get dismissed system notification keys
        $dismissedSystemKeys = \DB::table('dismissed_notifications')
            ->where('user_id', $user->id)
            ->pluck('notification_key')
            ->toArray();

        // Count unread persistent notifications (excluding deleted ones)
        $unreadDbNotifications = Notification::active()
            ->forParent($user->id, $childIds, $classIds)
            ->whereNotIn('id', $deletedNotificationIds)
            ->whereDoesntHave('reads', function($q) use ($user) {
                $q->where('user_id', $user->id)->whereNull('deleted_at');
            })
            ->count();

        // Count system notifications (excluding dismissed ones)
        $pendingFees = FeesCollection::whereIn('student_id', $childIds)
            ->where('status', 'pending')
            ->get()
            ->filter(function($fee) use ($dismissedSystemKeys) {
                return !in_array('fee_' . $fee->id, $dismissedSystemKeys);
            })
            ->count();

        $upcomingExams = ExamSchedule::whereIn('class_id', $classIds)
            ->where('exam_date', '>=', now())
            ->where('exam_date', '<=', now()->addDays(7))
            ->get()
            ->filter(function($exam) use ($dismissedSystemKeys) {
                return !in_array('exam_' . $exam->id, $dismissedSystemKeys);
            })
            ->count();

        $total = $unreadDbNotifications + $pendingFees + $upcomingExams;

        return response()->json([
            'success' => true,
            'data' => [
                'unread_count' => $total,
                'admin_notifications' => $unreadDbNotifications,
                'pending_fees' => $pendingFees,
                'upcoming_exams' => $upcomingExams,
            ]
        ], 200);
    }

    /**
     * Mark notification as read
     */
    public function markNotificationRead(Request $request, $id)
    {
        $user = $request->user();

        $notification = Notification::find($id);
        if ($notification) {
            $notification->markAsReadBy($user->id);
        }

        return response()->json([
            'success' => true,
            'message' => 'Notification marked as read'
        ], 200);
    }

    /**
     * Mark all notifications as read
     */
    public function markAllNotificationsRead(Request $request)
    {
        $user = $request->user();
        $children = $user->students;
        
        $childIds = $children->pluck('id')->toArray();
        $classIds = $children->pluck('class_id')->filter()->unique()->toArray();

        $notifications = Notification::active()
            ->forParent($user->id, $childIds, $classIds)
            ->get();

        foreach ($notifications as $notification) {
            $notification->markAsReadBy($user->id);
        }

        return response()->json([
            'success' => true,
            'message' => 'All notifications marked as read'
        ], 200);
    }

    /**
     * Delete a notification for the user
     */
    public function deleteNotification(Request $request, $id)
    {
        $user = $request->user();

        // Check if this is a system notification (e.g., fee_1, exam_2, result_3)
        if (strpos($id, '_') !== false) {
            // System notification - store in dismissed_notifications table
            \DB::table('dismissed_notifications')
                ->updateOrInsert(
                    ['user_id' => $user->id, 'notification_key' => $id],
                    ['dismissed_at' => now()]
                );
            
            return response()->json([
                'success' => true,
                'message' => 'Notification dismissed'
            ], 200);
        }

        // Admin notification
        $notification = Notification::find($id);
        if ($notification) {
            // Check if this notification is relevant to the user
            $children = $user->students;
            $childIds = $children->pluck('id')->toArray();
            $classIds = $children->pluck('class_id')->filter()->unique()->toArray();

            $isRelevant = $notification->for_all_parents ||
                          $notification->parent_id == $user->id ||
                          in_array($notification->student_id, $childIds) ||
                          in_array($notification->class_id, $classIds);

            if ($isRelevant) {
                $notification->markAsReadBy($user->id);
                
                \DB::table('notification_reads')
                    ->updateOrInsert(
                        ['notification_id' => $notification->id, 'user_id' => $user->id],
                        ['read_at' => now(), 'deleted_at' => now()]
                    );
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Notification deleted'
        ], 200);
    }

    /**
     * Get detailed information about a specific student
     */
    public function studentDetails(Request $request, $id)
    {
        $user = $request->user();
        
        // Verify that this student belongs to the parent
        $student = $user->students()
            ->with(['schoolClass', 'section', 'batch', 'academicYear'])
            ->where('id', $id)
            ->first();
        
        if (!$student) {
            return response()->json([
                'success' => false,
                'message' => 'Student not found or access denied'
            ], 404);
        }

        // Get attendance statistics
        $totalDays = Attendance::where('student_id', $student->id)->count();
        $presentDays = Attendance::where('student_id', $student->id)->where('status', 'present')->count();
        $absentDays = Attendance::where('student_id', $student->id)->where('status', 'absent')->count();
        $lateDays = Attendance::where('student_id', $student->id)->where('status', 'late')->count();
        
        $attendancePercentage = $totalDays > 0 ? round(($presentDays / $totalDays) * 100, 1) : 0;

        // Get pending fees
        $pendingFees = FeesCollection::where('student_id', $student->id)
            ->where('status', 'pending')
            ->sum('amount');

        // Get recent exam results
        $recentResults = ExamResult::where('student_id', $student->id)
            ->with(['examSchedule.subject', 'examSchedule.examType'])
            ->orderByDesc('created_at')
            ->limit(5)
            ->get()
            ->map(function ($result) {
                return [
                    'subject' => $result->examSchedule->subject->name ?? 'N/A',
                    'exam' => $result->examSchedule->examType->name ?? 'N/A',
                    'marks_obtained' => $result->marks_obtained,
                    'total_marks' => $result->examSchedule->max_marks ?? 100,
                    'grade' => $result->grade,
                    'date' => $result->created_at->format('d M Y'),
                ];
            });

        // Get pending homework count
        $pendingHomework = Homework::where('class_id', $student->class_id)
            ->where('section_id', $student->section_id)
            ->where('is_published', true)
            ->where('due_date', '>=', now())
            ->whereDoesntHave('submissions', function ($query) use ($student) {
                $query->where('student_id', $student->id)
                      ->where('status', 'submitted');
            })
            ->count();

        return response()->json([
            'success' => true,
            'data' => [
                'id' => $student->id,
                'admission_no' => $student->admission_no,
                'roll_number' => $student->roll_number,
                'name' => $student->name,
                'email' => $student->email,
                'phone' => $student->phone,
                'date_of_birth' => $student->date_of_birth ? $student->date_of_birth->format('d M Y') : null,
                'gender' => $student->gender,
                'blood_group' => $student->blood_group,
                'photo' => $student->photo,
                'class' => $student->schoolClass->name ?? null,
                'section' => $student->section->name ?? null,
                'batch' => $student->batch->name ?? null,
                'academic_year' => $student->academicYear->name ?? null,
                'address' => $student->address,
                'city' => $student->city,
                'state' => $student->state,
                'zip_code' => $student->zip_code,
                'father_name' => $student->father_name,
                'father_phone' => $student->father_phone,
                'father_occupation' => $student->father_occupation,
                'mother_name' => $student->mother_name,
                'mother_phone' => $student->mother_phone,
                'mother_occupation' => $student->mother_occupation,
                'guardian_name' => $student->guardian_name,
                'guardian_phone' => $student->guardian_phone,
                'guardian_relation' => $student->guardian_relation,
                'statistics' => [
                    'attendance_percentage' => $attendancePercentage,
                    'total_days' => $totalDays,
                    'present_days' => $presentDays,
                    'absent_days' => $absentDays,
                    'late_days' => $lateDays,
                    'pending_fees' => $pendingFees,
                    'pending_homework' => $pendingHomework,
                ],
                'recent_results' => $recentResults,
            ]
        ], 200);
    }

    /**
     * Get parent profile
     */
    public function profile(Request $request)
    {
        $user = $request->user();
        
        // Get children count
        $childrenCount = $user->students()->count();
        
        return response()->json([
            'success' => true,
            'data' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'phone' => $user->phone,
                'occupation' => $user->occupation,
                'address' => $user->address,
                'photo' => $user->photo,
                'role' => $user->role,
                'children_count' => $childrenCount,
                'created_at' => $user->created_at->format('d M Y'),
            ]
        ], 200);
    }

    /**
     * Update parent profile
     */
    public function updateProfile(Request $request)
    {
        $user = $request->user();
        
        $request->validate([
            'name' => 'sometimes|string|max:255',
            'phone' => 'sometimes|nullable|string|max:20',
            'occupation' => 'sometimes|nullable|string|max:255',
            'address' => 'sometimes|nullable|string|max:500',
            'photo' => 'sometimes|nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        // Update basic fields
        if ($request->has('name')) {
            $user->name = $request->name;
        }
        if ($request->has('phone')) {
            $user->phone = $request->phone;
        }
        if ($request->has('occupation')) {
            $user->occupation = $request->occupation;
        }
        if ($request->has('address')) {
            $user->address = $request->address;
        }

        // Handle photo upload
        if ($request->hasFile('photo')) {
            // Delete old photo if exists
            if ($user->photo && \Storage::disk('public')->exists($user->photo)) {
                \Storage::disk('public')->delete($user->photo);
            }
            
            $photo = $request->file('photo');
            $path = $photo->store('parents/photos', 'public');
            $user->photo = $path;
        }

        $user->save();

        return response()->json([
            'success' => true,
            'message' => 'Profile updated successfully',
            'data' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'phone' => $user->phone,
                'occupation' => $user->occupation,
                'address' => $user->address,
                'photo' => $user->photo,
            ]
        ], 200);
    }

    /**
     * Change password
     */
    public function changePassword(Request $request)
    {
        $user = $request->user();
        
        $request->validate([
            'current_password' => 'required|string',
            'new_password' => 'required|string|min:6|confirmed',
        ]);

        // Check current password
        if (!\Hash::check($request->current_password, $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Current password is incorrect'
            ], 400);
        }

        // Update password
        $user->password = \Hash::make($request->new_password);
        $user->save();

        return response()->json([
            'success' => true,
            'message' => 'Password changed successfully'
        ], 200);
    }

    /**
     * Register device token for push notifications
     */
    public function registerDeviceToken(Request $request)
    {
        $user = $request->user();
        
        $request->validate([
            'token' => 'required|string|max:500',
            'device_type' => 'sometimes|string|in:android,ios,web',
            'device_name' => 'nullable|string|max:255',
        ]);

        try {
            DeviceToken::registerToken(
                $user->id,
                $request->token,
                $request->device_type ?? 'android',
                $request->device_name
            );

            return response()->json([
                'success' => true,
                'message' => 'Device token registered successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to register device token'
            ], 500);
        }
    }

    /**
     * Unregister device token (logout from push notifications)
     */
    public function unregisterDeviceToken(Request $request)
    {
        $user = $request->user();
        
        $request->validate([
            'token' => 'required|string',
        ]);

        try {
            DeviceToken::where('user_id', $user->id)
                ->where('token', $request->token)
                ->update(['is_active' => false]);

            return response()->json([
                'success' => true,
                'message' => 'Device token unregistered successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to unregister device token'
            ], 500);
        }
    }
}
