<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Timetable;
use App\Models\SchoolClass;
use App\Models\Section;
use App\Models\Subject;
use App\Models\Teacher;
use App\Models\Room;

class TimetableController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        $sections = Section::where('is_active', true)->orderBy('name')->get();

        $classId = request('class_id');
        $sectionId = request('section_id');
        $day = request('day');

        $query = Timetable::with(['schoolClass', 'section', 'subject', 'teacher'])
            ->orderBy('day_of_week')
            ->orderBy('start_time');

        if ($classId) {
            $query->where('class_id', $classId);
        }
        if ($sectionId) {
            $query->where('section_id', $sectionId);
        }
        if ($day) {
            $query->where('day_of_week', $day);
        }

        $timetables = $query->get();

        $days = ['monday','tuesday','wednesday','thursday','friday','saturday','sunday'];

        return view('admin.timetable.index', compact('timetables', 'classes', 'sections', 'days', 'classId', 'sectionId', 'day', 'user'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        $sections = Section::where('is_active', true)->orderBy('name')->get();
        $subjects = Subject::where('is_active', true)->orderBy('name')->get();
        $teachers = Teacher::with('subjects')->orderBy('name')->get();
        $rooms = Room::where('is_active', true)->orderBy('name')->get();
        $days = ['monday','tuesday','wednesday','thursday','friday','saturday','sunday'];
        
        // Get all existing timetables
        $timetables = Timetable::with(['schoolClass', 'section', 'subject', 'teacher'])
            ->orderBy('day_of_week')
            ->orderBy('start_time')
            ->get();

        return view('admin.timetable.create', compact('classes', 'sections', 'subjects', 'teachers', 'rooms', 'days', 'user', 'timetables'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'class_id' => 'required|exists:school_classes,id',
            'section_id' => 'required|exists:sections,id',
            'subject_id' => 'required|exists:subjects,id',
            'teacher_id' => 'required|exists:teachers,id',
            'day_of_week' => 'required|in:monday,tuesday,wednesday,thursday,friday,saturday,sunday',
            'start_time' => 'required|date_format:H:i',
            'end_time' => 'required|date_format:H:i|after:start_time',
            'room' => 'nullable|string|max:255',
            'room_id' => 'nullable|exists:rooms,id',
        ]);

        // Check for teacher conflict: same teacher, same day, overlapping time
        $conflict = Timetable::where('teacher_id', $validated['teacher_id'])
            ->where('day_of_week', $validated['day_of_week'])
            ->where(function ($q) use ($validated) {
                $start = $validated['start_time'];
                $end = $validated['end_time'];

                $q->whereBetween('start_time', [$start, $end])
                  ->orWhereBetween('end_time', [$start, $end])
                  ->orWhere(function ($qq) use ($start, $end) {
                      $qq->where('start_time', '<=', $start)
                         ->where('end_time', '>=', $end);
                  });
            })
            ->exists();

        if ($conflict) {
            return back()
                ->withInput()
                ->withErrors(['teacher_id' => 'This teacher already has a class at the selected time on this day.']);
        }

        // If a room is selected from master, copy its name into room field
        if (!empty($validated['room_id'])) {
            $room = Room::find($validated['room_id']);
            if ($room) {
                $validated['room'] = $room->name;
            }
        }
        unset($validated['room_id']);

        Timetable::create($validated);

        return redirect()->route('admin.timetable.create')
            ->with('success', 'Timetable slot created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $timetable = Timetable::findOrFail($id);

        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        $sections = Section::where('is_active', true)->orderBy('name')->get();
        $subjects = Subject::where('is_active', true)->orderBy('name')->get();
        $teachers = Teacher::with('subjects')->orderBy('name')->get();
        $rooms = Room::where('is_active', true)->orderBy('name')->get();
        $days = ['monday','tuesday','wednesday','thursday','friday','saturday','sunday'];

        return view('admin.timetable.edit', compact('timetable', 'classes', 'sections', 'subjects', 'teachers', 'rooms', 'days', 'user'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $timetable = Timetable::findOrFail($id);

        $validated = $request->validate([
            'class_id' => 'required|exists:school_classes,id',
            'section_id' => 'required|exists:sections,id',
            'subject_id' => 'required|exists:subjects,id',
            'teacher_id' => 'required|exists:teachers,id',
            'day_of_week' => 'required|in:monday,tuesday,wednesday,thursday,friday,saturday,sunday',
            'start_time' => 'required|date_format:H:i',
            'end_time' => 'required|date_format:H:i|after:start_time',
            'room' => 'nullable|string|max:255',
            'room_id' => 'nullable|exists:rooms,id',
        ]);

        // Check for teacher conflict: same teacher, same day, overlapping time, excluding this slot
        $conflict = Timetable::where('teacher_id', $validated['teacher_id'])
            ->where('day_of_week', $validated['day_of_week'])
            ->where('id', '!=', $timetable->id)
            ->where(function ($q) use ($validated) {
                $start = $validated['start_time'];
                $end = $validated['end_time'];

                $q->whereBetween('start_time', [$start, $end])
                  ->orWhereBetween('end_time', [$start, $end])
                  ->orWhere(function ($qq) use ($start, $end) {
                      $qq->where('start_time', '<=', $start)
                         ->where('end_time', '>=', $end);
                  });
            })
            ->exists();

        if ($conflict) {
            return back()
                ->withInput()
                ->withErrors(['teacher_id' => 'This teacher already has a class at the selected time on this day.']);
        }

        if (!empty($validated['room_id'])) {
            $room = Room::find($validated['room_id']);
            if ($room) {
                $validated['room'] = $room->name;
            }
        }
        unset($validated['room_id']);

        $timetable->update($validated);

        return redirect()->route('admin.timetable.index')
            ->with('success', 'Timetable slot updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $timetable = Timetable::findOrFail($id);
        $timetable->delete();

        return redirect()->route('admin.timetable.index')
            ->with('success', 'Timetable slot deleted successfully.');
    }
}
