<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use App\Models\Setting;

class SettingController extends Controller
{
    public function edit()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        // Backwards compatibility: send to branding settings
        return redirect()->route('admin.settings.branding.edit');
    }

    public function editBranding()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $setting = Setting::first();

        return view('admin.settings.branding', compact('setting', 'user'));
    }

    public function update(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $validated = $request->validate([
            'site_name' => 'required|string|max:255',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'logo_light' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'logo_icon' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'login_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:4096',
            'contact_phone' => 'nullable|string|max:20',
            'contact_email' => 'nullable|email|max:255',
            'contact_whatsapp' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:500',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'zip_code' => 'nullable|string|max:20',
            'map_url' => 'nullable|url|max:500',
            'office_hours' => 'nullable|string|max:255',
            'website_url' => 'nullable|url|max:255',
            'facebook_url' => 'nullable|url|max:255',
            'instagram_url' => 'nullable|url|max:255',
        ]);

        $setting = Setting::first() ?? new Setting();
        $setting->site_name = $validated['site_name'];

        // Contact Info
        $setting->contact_phone = $validated['contact_phone'] ?? null;
        $setting->contact_email = $validated['contact_email'] ?? null;
        $setting->contact_whatsapp = $validated['contact_whatsapp'] ?? null;
        $setting->address = $validated['address'] ?? null;
        $setting->city = $validated['city'] ?? null;
        $setting->state = $validated['state'] ?? null;
        $setting->zip_code = $validated['zip_code'] ?? null;
        $setting->map_url = $validated['map_url'] ?? null;
        $setting->office_hours = $validated['office_hours'] ?? null;
        $setting->website_url = $validated['website_url'] ?? null;
        $setting->facebook_url = $validated['facebook_url'] ?? null;
        $setting->instagram_url = $validated['instagram_url'] ?? null;

        foreach (['logo', 'logo_light', 'logo_icon', 'login_image'] as $field) {
            if ($request->hasFile($field)) {
                // delete old file
                if ($setting->{$field}) {
                    Storage::disk('public')->delete($setting->{$field});
                }
                $setting->{$field} = $request->file($field)->store('settings', 'public');
            }
        }

        $setting->save();

        return redirect()->route('admin.settings.branding.edit')->with('success', 'Branding updated successfully.');
    }

    public function updateBranding(Request $request)
    {
        return $this->update($request);
    }

    public function editGateways()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $setting = Setting::first();

        return view('admin.settings.payment-gateways', compact('setting', 'user'));
    }

    public function updateGateways(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $validated = $request->validate([
            'cashfree_active' => 'sometimes|boolean',
            'cashfree_app_id' => 'nullable|string|max:255',
            'cashfree_secret_key' => 'nullable|string|max:255',
            'cashfree_mode' => 'nullable|string|max:50',
            'razorpay_active' => 'sometimes|boolean',
            'razorpay_key_id' => 'nullable|string|max:255',
            'razorpay_key_secret' => 'nullable|string|max:255',
            'phonepe_active' => 'sometimes|boolean',
            'phonepe_merchant_id' => 'nullable|string|max:255',
            'phonepe_salt_key' => 'nullable|string|max:255',
            'phonepe_salt_index' => 'nullable|string|max:50',
            'phonepe_mode' => 'nullable|string|max:50',
        ]);

        $setting = Setting::first() ?? new Setting();

        // gateway toggles
        $setting->cashfree_active = $request->boolean('cashfree_active');
        $setting->cashfree_app_id = $validated['cashfree_app_id'] ?? null;
        $setting->cashfree_secret_key = $validated['cashfree_secret_key'] ?? null;
        $setting->cashfree_mode = $validated['cashfree_mode'] ?? null;

        $setting->razorpay_active = $request->boolean('razorpay_active');
        $setting->razorpay_key_id = $validated['razorpay_key_id'] ?? null;
        $setting->razorpay_key_secret = $validated['razorpay_key_secret'] ?? null;

        $setting->phonepe_active = $request->boolean('phonepe_active');
        $setting->phonepe_merchant_id = $validated['phonepe_merchant_id'] ?? null;
        $setting->phonepe_salt_key = $validated['phonepe_salt_key'] ?? null;
        $setting->phonepe_salt_index = $validated['phonepe_salt_index'] ?? null;
        $setting->phonepe_mode = $validated['phonepe_mode'] ?? null;

        $setting->save();

        return redirect()->route('admin.settings.gateways.edit')->with('success', 'Payment gateways updated successfully.');
    }

    public function editNotifications()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $setting = Setting::first();

        return view('admin.settings.notifications', compact('setting', 'user'));
    }

    public function updateNotifications(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $validated = $request->validate([
            'fcm_project_id' => 'nullable|string|max:255',
            'service_account_file' => 'nullable|file|mimes:json|max:100',
            'push_notifications_enabled' => 'sometimes|boolean',
        ]);

        $setting = Setting::first() ?? new Setting();
        $setting->fcm_project_id = $validated['fcm_project_id'] ?? null;
        $setting->push_notifications_enabled = $request->boolean('push_notifications_enabled');

        // Handle service account JSON file upload
        if ($request->hasFile('service_account_file')) {
            $file = $request->file('service_account_file');
            $contents = file_get_contents($file->getRealPath());
            $json = json_decode($contents, true);
            
            if ($json && isset($json['project_id'])) {
                // Auto-fill project ID from service account
                $setting->fcm_project_id = $json['project_id'];
                
                // Save the file to storage
                Storage::put('firebase-service-account.json', $contents);
                
                // Clear cached access token
                \Cache::forget('fcm_access_token');
            } else {
                return back()->withErrors(['service_account_file' => 'Invalid service account JSON file.'])->withInput();
            }
        }

        $setting->save();

        return redirect()->route('admin.settings.notifications.edit')->with('success', 'Push notification settings updated successfully.');
    }
}



