<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\User;

class ParentController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $search = request('search');

        $query = User::where('role', 'parent');

        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        $parents = $query->withCount('students')->orderBy('name')->get();

        return view('admin.parent.index', compact('parents', 'user', 'search'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        return view('admin.parent.create', compact('user'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:users,email',
            'password' => 'required|string|min:6',
            'phone' => 'nullable|string|max:20',
            'occupation' => 'nullable|string|max:255',
            'address' => 'nullable|string',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $photoPath = null;
        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('parents', 'public');
        }

        User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => bcrypt($validated['password']),
            'role' => 'parent',
            'phone' => $validated['phone'] ?? null,
            'occupation' => $validated['occupation'] ?? null,
            'address' => $validated['address'] ?? null,
            'photo' => $photoPath,
        ]);

        return redirect()->route('admin.parent.index')
            ->with('success', 'Parent created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $parent = User::where('role', 'parent')->with('students')->findOrFail($id);

        return view('admin.parent.show', compact('parent', 'user'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $parent = User::where('role', 'parent')->findOrFail($id);

        return view('admin.parent.edit', compact('parent', 'user'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $parent = User::where('role', 'parent')->findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:users,email,' . $id,
            'password' => 'nullable|string|min:6',
            'phone' => 'nullable|string|max:20',
            'occupation' => 'nullable|string|max:255',
            'address' => 'nullable|string',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $parent->name = $validated['name'];
        $parent->email = $validated['email'];
        if (!empty($validated['password'])) {
            $parent->password = bcrypt($validated['password']);
        }
        $parent->phone = $validated['phone'] ?? $parent->phone;
        $parent->occupation = $validated['occupation'] ?? $parent->occupation;
        $parent->address = $validated['address'] ?? $parent->address;

        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('parents', 'public');
            $parent->photo = $photoPath;
        }

        $parent->save();

        return redirect()->route('admin.parent.index')
            ->with('success', 'Parent updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $parent = User::where('role', 'parent')->findOrFail($id);

        // Optionally you might want to detach students or handle them differently
        // For now, we'll just set their parent_id to null
        foreach ($parent->students as $student) {
            $student->parent_id = null;
            $student->save();
        }

        $parent->delete();

        return redirect()->route('admin.parent.index')
            ->with('success', 'Parent deleted successfully.');
    }
}
