<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use App\Models\Notification;
use App\Models\SchoolClass;
use App\Models\Student;
use App\Models\User;
use App\Models\DeviceToken;
use App\Services\PushNotificationService;

class NotificationController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $notifications = Notification::with(['schoolClass', 'student', 'parent', 'createdBy'])
            ->orderByDesc('created_at')
            ->paginate(20);

        return view('admin.notifications.index', compact('notifications', 'user'));
    }

    public function create()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $classes = SchoolClass::orderBy('name')->get();
        $students = Student::orderBy('name')->get();
        $parents = User::where('role', 'parent')->orderBy('name')->get();
        $types = Notification::$types;

        return view('admin.notifications.create', compact('user', 'classes', 'students', 'parents', 'types'));
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'message' => 'required|string',
            'type' => 'required|string|in:general,fee,leave,exam,result,homework,attendance,announcement',
            'target' => 'required|string|in:all,class,student,parent',
            'class_id' => 'nullable|exists:school_classes,id',
            'student_id' => 'nullable|exists:students,id',
            'parent_id' => 'nullable|exists:users,id',
            'scheduled_at' => 'nullable|date',
            'expires_at' => 'nullable|date|after:scheduled_at',
        ]);

        $typeConfig = Notification::$types[$validated['type']] ?? ['icon' => 'info', 'color' => 'blue'];

        $notification = new Notification();
        $notification->title = $validated['title'];
        $notification->message = $validated['message'];
        $notification->type = $validated['type'];
        $notification->icon = $typeConfig['icon'];
        $notification->color = $typeConfig['color'];
        $notification->created_by = $user->id;
        $notification->scheduled_at = $validated['scheduled_at'] ?? null;
        $notification->expires_at = $validated['expires_at'] ?? null;

        // Set target
        switch ($validated['target']) {
            case 'all':
                $notification->for_all_parents = true;
                break;
            case 'class':
                $notification->class_id = $validated['class_id'];
                break;
            case 'student':
                $notification->student_id = $validated['student_id'];
                break;
            case 'parent':
                $notification->parent_id = $validated['parent_id'];
                break;
        }

        $notification->save();

        // Send push notification
        $this->sendPushNotification($notification, $validated['target'], $validated);

        return redirect()->route('admin.notifications.index')->with('success', 'Notification sent successfully!');
    }

    public function edit($id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $notification = Notification::findOrFail($id);
        $classes = SchoolClass::orderBy('name')->get();
        $students = Student::orderBy('name')->get();
        $parents = User::where('role', 'parent')->orderBy('name')->get();
        $types = Notification::$types;

        return view('admin.notifications.edit', compact('user', 'notification', 'classes', 'students', 'parents', 'types'));
    }

    public function update(Request $request, $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $notification = Notification::findOrFail($id);

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'message' => 'required|string',
            'type' => 'required|string|in:general,fee,leave,exam,result,homework,attendance,announcement',
            'target' => 'required|string|in:all,class,student,parent',
            'class_id' => 'nullable|exists:school_classes,id',
            'student_id' => 'nullable|exists:students,id',
            'parent_id' => 'nullable|exists:users,id',
            'scheduled_at' => 'nullable|date',
            'expires_at' => 'nullable|date',
            'is_active' => 'sometimes|boolean',
        ]);

        $typeConfig = Notification::$types[$validated['type']] ?? ['icon' => 'info', 'color' => 'blue'];

        $notification->title = $validated['title'];
        $notification->message = $validated['message'];
        $notification->type = $validated['type'];
        $notification->icon = $typeConfig['icon'];
        $notification->color = $typeConfig['color'];
        $notification->scheduled_at = $validated['scheduled_at'] ?? null;
        $notification->expires_at = $validated['expires_at'] ?? null;
        $notification->is_active = $request->boolean('is_active', true);

        // Reset targets
        $notification->for_all_parents = false;
        $notification->class_id = null;
        $notification->student_id = null;
        $notification->parent_id = null;

        // Set target
        switch ($validated['target']) {
            case 'all':
                $notification->for_all_parents = true;
                break;
            case 'class':
                $notification->class_id = $validated['class_id'];
                break;
            case 'student':
                $notification->student_id = $validated['student_id'];
                break;
            case 'parent':
                $notification->parent_id = $validated['parent_id'];
                break;
        }

        $notification->save();

        return redirect()->route('admin.notifications.index')->with('success', 'Notification updated successfully!');
    }

    public function destroy($id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $notification = Notification::findOrFail($id);
        $notification->delete();

        return redirect()->route('admin.notifications.index')->with('success', 'Notification deleted successfully!');
    }

    public function toggleStatus($id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $notification = Notification::findOrFail($id);
        $notification->is_active = !$notification->is_active;
        $notification->save();

        return redirect()->route('admin.notifications.index')->with('success', 'Notification status updated!');
    }

    /**
     * Send push notification to targeted devices
     */
    protected function sendPushNotification(Notification $notification, string $target, array $validated)
    {
        try {
            $pushService = new PushNotificationService();
            $tokens = [];
            $data = [
                'notification_id' => (string) $notification->id,
                'type' => $notification->type,
                'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
            ];

            switch ($target) {
                case 'all':
                    // Get all parent user IDs
                    $parentIds = User::where('role', 'parent')->pluck('id')->toArray();
                    $tokens = DeviceToken::whereIn('user_id', $parentIds)
                        ->where('is_active', true)
                        ->pluck('token')
                        ->toArray();
                    break;

                case 'class':
                    // Get parent IDs for students in this class
                    $parentIds = Student::where('class_id', $validated['class_id'])
                        ->whereNotNull('parent_id')
                        ->pluck('parent_id')
                        ->unique()
                        ->toArray();
                    $tokens = DeviceToken::whereIn('user_id', $parentIds)
                        ->where('is_active', true)
                        ->pluck('token')
                        ->toArray();
                    break;

                case 'student':
                    // Get parent ID for this student
                    $student = Student::find($validated['student_id']);
                    if ($student && $student->parent_id) {
                        $tokens = DeviceToken::where('user_id', $student->parent_id)
                            ->where('is_active', true)
                            ->pluck('token')
                            ->toArray();
                    }
                    break;

                case 'parent':
                    // Get tokens for specific parent
                    $tokens = DeviceToken::where('user_id', $validated['parent_id'])
                        ->where('is_active', true)
                        ->pluck('token')
                        ->toArray();
                    break;
            }

            if (!empty($tokens)) {
                Log::info('Sending push notification to ' . count($tokens) . ' devices');
                $pushService->sendToTokens($tokens, $notification->title, $notification->message, $data);
            } else {
                Log::info('No device tokens found for target: ' . $target);
            }
        } catch (\Exception $e) {
            Log::error('Error sending push notification: ' . $e->getMessage());
        }
    }
}
