<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Note;
use App\Models\SchoolClass;
use App\Services\PushNotificationService;

class NoteController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $notes = Note::with('schoolClass')->orderByDesc('created_at')->paginate(15);

        return view('admin.notes.index', compact('notes', 'user'));
    }

    public function create()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $classes = SchoolClass::orderBy('name')->get();
        return view('admin.notes.create', compact('classes', 'user'));
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'body' => 'nullable|string',
            'for_all' => 'sometimes|boolean',
            'class_id' => 'nullable|exists:school_classes,id',
        ]);

        // if not for all, class_id required
        $forAll = $request->boolean('for_all');
        if (!$forAll && empty($validated['class_id'])) {
            return back()->withErrors(['class_id' => 'Please select a class or choose All Classes.'])->withInput();
        }

        $note = Note::create([
            'title' => $validated['title'],
            'body' => $validated['body'] ?? null,
            'for_all' => $forAll,
            'class_id' => $forAll ? null : $validated['class_id'],
        ]);

        // Send push notification
        try {
            $pushService = new PushNotificationService();
            $pushService->notifyAnnouncement($note, $forAll, $validated['class_id'] ?? null);
        } catch (\Exception $e) {
            \Log::error('Failed to send announcement notification: ' . $e->getMessage());
        }

        return redirect()->route('admin.notes.index')->with('success', 'Note created successfully.');
    }

    public function edit(Note $note)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $classes = SchoolClass::orderBy('name')->get();
        return view('admin.notes.edit', compact('note', 'classes', 'user'));
    }

    public function update(Request $request, Note $note)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'body' => 'nullable|string',
            'for_all' => 'sometimes|boolean',
            'class_id' => 'nullable|exists:school_classes,id',
        ]);

        $forAll = $request->boolean('for_all');
        if (!$forAll && empty($validated['class_id'])) {
            return back()->withErrors(['class_id' => 'Please select a class or choose All Classes.'])->withInput();
        }

        $note->update([
            'title' => $validated['title'],
            'body' => $validated['body'] ?? null,
            'for_all' => $forAll,
            'class_id' => $forAll ? null : $validated['class_id'],
        ]);

        return redirect()->route('admin.notes.index')->with('success', 'Note updated successfully.');
    }

    public function destroy(Note $note)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $note->delete();

        return redirect()->route('admin.notes.index')->with('success', 'Note deleted successfully.');
    }
}


