<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Student;
use App\Models\Teacher;
use App\Models\User;
use App\Models\SchoolClass;
use App\Models\FeesCollection;
use App\Models\LeaveRequest;
use App\Models\Note;
use App\Models\Timetable;
use App\Models\ExamResult;
use App\Models\ExamSchedule;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    /**
     * Show the admin dashboard
     */
    public function index()
    {
        $user = Auth::user();
        
        // Ensure user is authenticated
        if (!$user) {
            return redirect()->route('login');
        }
        
        // Ensure only admin can access
        if (!$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $totalStudents = Student::count();
        $totalTeachers = Teacher::count();
        $totalParents = User::where('role', 'parent')->count();
        $totalClasses = SchoolClass::count();

        $today = Carbon::today();
        $monthStart = $today->copy()->startOfMonth();
        $lastMonthStart = $today->copy()->subMonth()->startOfMonth();
        $lastMonthEnd = $today->copy()->subMonth()->endOfMonth();

        // Fee statistics
        $feesToday = FeesCollection::whereDate('payment_date', $today)->sum('paid_amount');
        $feesThisMonth = FeesCollection::whereBetween('payment_date', [$monthStart, $today])->sum('paid_amount');
        $feesLastMonth = FeesCollection::whereBetween('payment_date', [$lastMonthStart, $lastMonthEnd])->sum('paid_amount');
        
        // Total fees collected (all time)
        $totalFeesCollected = FeesCollection::sum('paid_amount');
        
        // Pending fees
        $totalFeesAmount = FeesCollection::sum('amount');
        $pendingFees = $totalFeesAmount - $totalFeesCollected;
        
        // This month new students
        $newStudentsThisMonth = Student::where('created_at', '>=', $monthStart)->count();
        $newStudentsLastMonth = Student::whereBetween('created_at', [$lastMonthStart, $lastMonthEnd])->count();
        
        // Calculate growth percentages
        $feeGrowth = $feesLastMonth > 0 ? round((($feesThisMonth - $feesLastMonth) / $feesLastMonth) * 100, 1) : 0;
        $studentGrowth = $newStudentsLastMonth > 0 ? round((($newStudentsThisMonth - $newStudentsLastMonth) / $newStudentsLastMonth) * 100, 1) : 0;

        $pendingLeaves = LeaveRequest::where('status', 'pending')->count();

        $recentLeaves = LeaveRequest::with(['user','student','leaveType'])
            ->orderByDesc('created_at')
            ->limit(5)
            ->get();

        $latestNotes = Note::orderByDesc('created_at')->limit(5)->get();

        // Current and upcoming sessions (today)
        $now = Carbon::now();
        $dayKey = strtolower($now->format('l'));
        $timeNow = $now->format('H:i:s');

        $currentSessions = Timetable::with(['schoolClass','section','subject','teacher'])
            ->where('day_of_week', $dayKey)
            ->where('start_time', '<=', $timeNow)
            ->where('end_time', '>', $timeNow)
            ->orderBy('start_time')
            ->limit(5)
            ->get();

        $upcomingSessions = Timetable::with(['schoolClass','section','subject','teacher'])
            ->where('day_of_week', $dayKey)
            ->where('start_time', '>', $timeNow)
            ->orderBy('start_time')
            ->limit(5)
            ->get();

        // Top students from last exam
        $topStudents = collect();
        $lastExam = ExamSchedule::with('examType')
            ->whereHas('results')
            ->orderByDesc('exam_date')
            ->first();
        
        if ($lastExam) {
            $topStudents = ExamResult::with(['student.schoolClass', 'examSchedule.examType', 'examSchedule.subject'])
                ->where('exam_schedule_id', $lastExam->id)
                ->where('is_absent', false)
                ->orderByDesc('marks_obtained')
                ->limit(10)
                ->get()
                ->map(function ($result) {
                    $maxMarks = $result->examSchedule->max_marks ?? 100;
                    $marksObtained = $result->marks_obtained ?? 0;
                    $percentage = $maxMarks > 0 ? round(($marksObtained / $maxMarks) * 100, 1) : 0;
                    
                    return [
                        'student' => $result->student,
                        'marks' => $marksObtained,
                        'total' => $maxMarks,
                        'percentage' => $percentage,
                        'exam_name' => $result->examSchedule->examType->name ?? 'Exam',
                        'subject' => $result->examSchedule->subject->name ?? 'Subject',
                    ];
                });
        }

        return view('admin.dashboard', [
            'user' => $user,
            'totalStudents' => $totalStudents,
            'totalTeachers' => $totalTeachers,
            'totalParents' => $totalParents,
            'totalClasses' => $totalClasses,
            'feesToday' => $feesToday,
            'feesThisMonth' => $feesThisMonth,
            'feesLastMonth' => $feesLastMonth,
            'totalFeesCollected' => $totalFeesCollected,
            'pendingFees' => $pendingFees,
            'feeGrowth' => $feeGrowth,
            'studentGrowth' => $studentGrowth,
            'newStudentsThisMonth' => $newStudentsThisMonth,
            'pendingLeaves' => $pendingLeaves,
            'recentLeaves' => $recentLeaves,
            'latestNotes' => $latestNotes,
            'currentSessions' => $currentSessions,
            'upcomingSessions' => $upcomingSessions,
            'topStudents' => $topStudents,
            'lastExam' => $lastExam,
        ]);
    }
}
